#include "animation_chronology.h"

#include <yandex_io/libs/logging/logging.h>

#include <map>

YIO_DEFINE_LOG_MODULE("leds");

void AnimationChronology::updateTime(TimePoint timePoint) {
    animationBackground.updateTime(timePoint);
    updateCompositions();
}

void AnimationChronology::updateCompositions() {
    auto currentComposition = animationBackground.currentCompositions();
    if (differ(currentAnimationCompositions_, currentComposition)) {
        animationBackground.logCurrentCompositions();
        shift(std::move(currentComposition));
        hasNew_ = true;

        YIO_LOG_DEBUG("updateCompositions, len of currentAnimations = " << getAnimations().size());
    } else {
        hasNew_ = false;
    }
}

std::vector<std::shared_ptr<Animation>> AnimationChronology::getAnimations() {
    std::map<std::shared_ptr<quasar::LedDevice>, std::shared_ptr<Animation>,
             std::owner_less<std::shared_ptr<quasar::LedDevice>>>
        animationsPerDevice; // TODO: optimize? cache?
    for (const auto& animationComposition : currentAnimationCompositions_) {
        if (!animationComposition) {
            YIO_LOG_ERROR_EVENT("AnimationChronology.NullComposition", "composition is null");
            continue;
        }
        for (const auto& animation : animationComposition->getAnimations()) {
            auto device = animation->getDevice().lock();
            animationsPerDevice.insert(std::pair(device, animation));
        }
    }

    std::vector<std::shared_ptr<Animation>> result;
    for (const auto& entry : animationsPerDevice) {
        result.emplace_back(entry.second);
    }
    return result;
}

void AnimationChronology::shift(std::vector<const std::shared_ptr<AnimationComposition>> newCurrentAnimation) {
    // if we need to save any information about previous animations, this is a point designed for it
    currentAnimationCompositions_ = std::move(newCurrentAnimation);
}

bool AnimationChronology::hasNew() const {
    return hasNew_;
}

// makes a reasonable assumption, that v1 contains no repeating elements
bool AnimationChronology::differ(const std::vector<const std::shared_ptr<AnimationComposition>>& v1,
                                 const std::vector<const std::shared_ptr<AnimationComposition>>& v2) {
    if (v1.size() != v2.size()) {
        return true;
    }

    for (const auto& first : v1) {
        bool found = false;
        for (const auto& second : v2) {
            if (first == second) {
                found = true;
                break;
            }
        }
        if (!found) {
            return true;
        }
    }
    return false;
}

void AnimationChronology::playNow(std::shared_ptr<AnimationConductor> animationConductor) {
    if (animationConductor->getSubstitutionType() == SubstitutionType::BACKGROUND_DEFAULT) {
        animationBackground.setBackgroundAnimation(std::move(animationConductor));
    } else if (animationConductor->getSubstitutionType() == SubstitutionType::FOREGROUND) {
        animationBackground.setForegroundAnimation(std::move(animationConductor));
    } else if (animationConductor->getSubstitutionType() == SubstitutionType::SUBSTITUTION) {
        animationBackground.setSubstitutionAnimation(std::move(animationConductor));
    }
    updateCompositions();
    hasNew_ = true;
}

void AnimationChronology::resetNew() {
    hasNew_ = false;
}

void AnimationChronology::removeForeground() {
    animationBackground.removeForeground();
    updateCompositions();
    hasNew_ = true;
}
