#include "animation_conductor.h"

#include <algorithm>

AnimationConductor::AnimationConductor(std::vector<const std::shared_ptr<AnimationComposition>> compositions,
                                       SubstitutionType substitutionType, FinishingType finishingType, std::string name)
    : substitutionType_(substitutionType)
    , finishingType_(finishingType)
    , animationCompositionList_(std::move(compositions))
    , name_(name)
{
    resetConductor();
}

std::string AnimationConductor::getName() const {
    return name_;
}

const std::shared_ptr<AnimationComposition>& AnimationConductor::nextComposition() {
    currentAnimationComposition_++;
    if (onFinishedListener_ && isFinished()) {
        onFinishedListener_();
    }
    return *currentAnimationComposition_;
}

std::shared_ptr<AnimationComposition> AnimationConductor::getCurrentComposition() const {
    return *currentAnimationComposition_;
}

void AnimationConductor::resetConductor() {
    currentAnimationComposition_ = animationCompositionList_.begin();
}

bool AnimationConductor::isFinished() const {
    if (animationCompositionList_.size() == 0) {
        return true;
    }
    return currentAnimationComposition_ == animationCompositionList_.end();
}

bool AnimationConductor::updateTime(AnimationConductor::TimePoint timePoint) {
    if (isFinished()) {
        return true;
    }
    bool finished = false;
    const std::shared_ptr<AnimationComposition>& currentComposition = getCurrentComposition();

    const auto& animations = currentComposition->getAnimations();
    for (const auto& animation : animations) {
        if (!animation->finished()) {
            animation->updateTime(timePoint);
        }
    }

    if (finishingType_ == FinishingType::FIRST) {
        finished = animations.empty() || animations[0]->finished();
    } else {
        finished = std::all_of(animations.cbegin(), animations.cend(),
                               [](const auto& animation) { return animation->finished(); });
    }
    if (finished) {
        if (currentComposition->isLooped()) {
            currentComposition->restartComposition();
        } else {
            nextComposition();
        }
    }
    return false;
}

SubstitutionType AnimationConductor::getSubstitutionType() {
    return substitutionType_;
}

void AnimationConductor::setOnFinishedListener(AnimationConductor::OnFinishedListener onFinishedListener) {
    onFinishedListener_ = std::move(onFinishedListener);
}

void AnimationConductor::setOnChangedListener(AnimationConductor::OnChangedListener onChangedListener) {
    onChangedListener_ = std::move(onChangedListener);
}
