#include "clock_animation_base.h"

#include <yandex_io/libs/errno/errno_exception.h>

#include <chrono>
#include <ctime>

namespace {

    std::string getTimeString(int64_t utcOffsetSeconds) {
        std::time_t t;
        std::time(&t);
        t += utcOffsetSeconds;

        struct tm timeInfo;
        if (localtime_r(&t, &timeInfo) == nullptr) {
            throw quasar::ErrnoException(errno, "Can't get localtime");
        }

        std::string timeString;
        timeString.reserve(5);

        timeString += static_cast<char>('0' + timeInfo.tm_hour / 10);
        timeString += static_cast<char>('0' + timeInfo.tm_hour % 10);
        timeString += ':';
        timeString += static_cast<char>('0' + timeInfo.tm_min / 10);
        timeString += static_cast<char>('0' + timeInfo.tm_min % 10);

        return timeString;
    }

} // namespace

ClockAnimationBase::ClockAnimationBase(std::weak_ptr<quasar::LedDevice> ledDevice, int64_t utcOffsetSeconds)
    : Animation(ledDevice)
    , utcOffsetSeconds_(utcOffsetSeconds)
{
    updateTimeStr();
}

bool ClockAnimationBase::updateTimeStr() {
    const std::string timeStr = getTimeString(utcOffsetSeconds_);

    if (timeStr_ == timeStr) {
        return false;
    }

    timeStr_ = timeStr;

    return true;
}

bool ClockAnimationBase::finished() const {
    return false;
}

Animation::TimePoint ClockAnimationBase::getEndOfFrameTimePoint() const {
    return endOfFrame_;
}

std::string ClockAnimationBase::getName() const {
    return "clock";
}

void ClockAnimationBase::resetAnimation() {
    updateTimeStr();

    endOfFrame_ = std::chrono::steady_clock::now() + std::chrono::seconds(1);
    started_ = false;
}

void ClockAnimationBase::updateTime(Animation::TimePoint timePoint) {
    if (!started_) {
        return;
    }

    while (timePoint >= endOfFrame_) {
        endOfFrame_ = endOfFrame_ + frameLength_;
        updateTimeStr();
    }
}

void ClockAnimationBase::startAnimationFrom(Animation::TimePoint timePoint) {
    if (started_) {
        return;
    }

    updateTimeStr();
    endOfFrame_ = timePoint + frameLength_;
    started_ = true;
}

std::chrono::nanoseconds ClockAnimationBase::getLength() const {
    return frameLength_;
}

bool ClockAnimationBase::started() const {
    return started_;
}
