#include "timer_countdown_animation_base.h"

#include <chrono>

namespace {
    std::string toStingWithZero(uint64_t val) {
        if (val < 10) {
            return "0" + std::to_string(val);
        } else {
            return std::to_string(val);
        }
    }
} // namespace

TimerCountdownAnimationBase::TimerCountdownAnimationBase(std::weak_ptr<quasar::LedDevice> ledDevice,
                                                         std::chrono::system_clock::time_point timerEnd)
    : Animation(ledDevice)
    , timerEnd_(timerEnd)
{
    updateTimeStr();
}

std::string TimerCountdownAnimationBase::getName() const {
    return "timer_countdown";
}

void TimerCountdownAnimationBase::updateTimeStr() {
    auto now = std::chrono::system_clock::now();

    if (now > timerEnd_) {
        if (timeStr_ == "00:00") {
            finished_ = true;
            return;
        }

        timeStr_ = "00:00";
        return;
    }

    auto remainingTime = std::chrono::duration_cast<std::chrono::seconds>(timerEnd_ - now);
    auto minutes = remainingTime.count() / 60;
    auto seconds = remainingTime.count() % 60;
    if (minutes > 99) {
        throw std::runtime_error("Remaining timer time is too big");
    }

    const std::string timeStr = toStingWithZero(minutes) + ":" + toStingWithZero(seconds);
    if (timeStr_ == timeStr) {
        return;
    }

    timeStr_ = timeStr;
}

Animation::TimePoint TimerCountdownAnimationBase::getEndOfFrameTimePoint() const {
    return endOfFrame_;
}

void TimerCountdownAnimationBase::resetAnimation() {
    updateTimeStr();

    endOfFrame_ = std::chrono::steady_clock::now() + std::chrono::seconds(1);
    started_ = false;
}

std::chrono::nanoseconds TimerCountdownAnimationBase::getLength() const {
    return frameLength_;
}

bool TimerCountdownAnimationBase::started() const {
    return started_;
}

void TimerCountdownAnimationBase::startAnimationFrom(TimePoint timePoint) {
    if (started()) {
        return;
    }

    updateTimeStr();
    endOfFrame_ = timePoint + frameLength_;
    started_ = true;
}

bool TimerCountdownAnimationBase::finished() const {
    return finished_;
}

void TimerCountdownAnimationBase::updateTime(TimePoint timePoint) {
    if (!started_) {
        return;
    }

    while (timePoint >= endOfFrame_) {
        endOfFrame_ = endOfFrame_ + frameLength_;
        updateTimeStr();
    }
}
