#include "led_pattern.h"

#include <yandex_io/libs/logging/logging.h>

#include <boost/algorithm/string.hpp>

#include <util/system/yassert.h>

#include <algorithm>
#include <cmath>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <stdexcept>

using namespace quasar;

std::shared_ptr<LedPattern> LedPattern::loadFromFile(const std::string& fileName,
                                                     const std::weak_ptr<quasar::LedController>& ledControllerWeak)
{
    auto ledController = ledControllerWeak.lock();
    if (!ledController) {
        throw std::runtime_error("no LedController passed");
    }
    if (ledController->getHeight() != 1) {
        throw std::runtime_error("led pattern is only for 1 pixel height");
    }
    int ledCount = ledController->getWidth();
    std::ifstream patternFile(fileName);
    auto pattern = std::make_shared<LedPattern>(ledCount, fileName, ledControllerWeak);

    if (patternFile.is_open())
    {
        try {
            pattern->load(patternFile);
        } catch (const std::exception& e)
        {
            throw std::runtime_error("Cannot load LED pattern from file '" + fileName + "': " + e.what());
        }
    } else {
        throw std::runtime_error("Cannot open LED pattern file '" + fileName + "'");
    }

    return pattern;
}

LedPattern::LedPattern(int ledCount, std::weak_ptr<quasar::LedController> ledController)
    : Animation(ledController)
    , ledCount_(ledCount)
    , ledDevice_(std::move(ledController))
{
}

LedPattern::LedPattern(int ledCount,
                       std::string name,
                       std::weak_ptr<quasar::LedController> ledController)
    : Animation(ledController)
    , ledCount_(ledCount)
    , name_(std::move(name))
    , ledDevice_(std::move(ledController))
{
}

std::weak_ptr<quasar::LedController> LedPattern::getDevice() {
    return ledDevice_;
}

void LedPattern::load(std::istream& stream)
{
    std::string tmp;
    std::string line;
    LedPattern& pattern = *this;
    while (std::getline(stream, line))
    {
        boost::trim_if(line, boost::is_any_of(" \t"));
        if (line.empty() || line[0] == '#') {
            continue;
        }

        std::stringstream lineStream(line);
        if (boost::istarts_with(line, "no_size_optimization"))
        {
            optimizeSize_ = false;
            continue;
        }
        if (boost::istarts_with(line, "loop"))
        {
            pattern.loopFromIndex = int(pattern.frames.size());
            continue;
        }
        if (boost::istarts_with(line, "smooth"))
        {
            lineStream >> tmp >> smooth_;
            logSmooth_ = false;
            continue;
        } else if (boost::istarts_with(line, "logsmooth"))
        {
            lineStream >> tmp >> smooth_;
            logSmooth_ = true;
            continue;
        } else if (boost::istarts_with(line, "gammacorrection"))
        {
            lineStream >> tmp;
            gammaCorrection_ = true;
            continue;
        } else if (boost::istarts_with(line, "background"))
        {
            lineStream >> tmp;
            isBackground_ = true;
            continue;
        } else if (boost::istarts_with(line, "name"))
        {
            lineStream >> tmp >> name_;
            continue;
        }

        bool rotate = false;
        bool step = false;
        bool step2 = false;
        bool fill = false;
        bool unfill = false;
        if (boost::istarts_with(line, "rotate"))
        {
            lineStream >> tmp;
            rotate = true;
        } else if (boost::istarts_with(line, "step1"))
        {
            lineStream >> tmp;
            step = true;
        } else if (boost::istarts_with(line, "step2"))
        {
            lineStream >> tmp;
            step2 = true;
        } else if (boost::istarts_with(line, "fill"))
        {
            lineStream >> tmp;
            fill = true;
        } else if (boost::istarts_with(line, "unfill"))
        {
            lineStream >> tmp;
            unfill = true;
        }
        LedFrame ledFrame = getFrameFromStream(lineStream, ledCount_);
        if (loopFromIndex == int(frames.size())) {
            firstLoopFrameDelayMs_ = ledFrame.delayMs;
        }

        if (rotate)
        {
            for (int i = 0; i < ledCount_; ++i)
            {
                addSmoothFrame(getRotatedFrame(ledFrame, i, ledCount_));
            }
        } else if (step)
        {
            LedFrame frame = ledFrame;
            for (int i = 0; i < ledCount_; ++i)
            {
                std::fill(frame.circle.begin(), frame.circle.end(), rgbw_color());
                std::copy(ledFrame.circle.begin(), ledFrame.circle.begin() + i + 1, frame.circle.begin());
                addSmoothFrame(getRotatedFrame(frame, i - 1, ledCount_));
            }
            addSmoothFrame(ledFrame);
            for (int i = 0; i < ledCount_ - 1; ++i)
            {
                std::fill(frame.circle.begin(), frame.circle.end(), rgbw_color());
                std::copy(ledFrame.circle.begin(), ledFrame.circle.begin() + ledCount_ - i - 1, frame.circle.begin() + i + 1);
                addSmoothFrame(getRotatedFrame(frame, i + 1, ledCount_));
            }

        } else if (step2)
        {
            LedFrame frame = ledFrame;
            for (int i = 1; i <= ledCount_; ++i)
            {
                std::fill(frame.circle.begin(), frame.circle.end(), rgbw_color());
                std::copy(ledFrame.circle.begin(), ledFrame.circle.begin() + i, frame.circle.begin());
                addSmoothFrame(frame);
            }
            for (int i = 1; i < ledCount_; ++i)
            {
                frame = ledFrame;
                std::fill(frame.circle.begin(), frame.circle.begin() + i, rgbw_color());
                addSmoothFrame(frame);
            }

        } else if (fill)
        {
            for (int i = 0; i < ledCount_; ++i)
            {
                LedFrame frame = ledFrame;
                std::fill(frame.circle.begin() + i + 1, frame.circle.end(), rgbw_color());
                addSmoothFrame(frame);
            }
        } else if (unfill) {
            for (int i = 0; i < ledCount_; ++i)
            {
                LedFrame frame = ledFrame;
                std::fill(frame.circle.begin(), frame.circle.begin() + i, rgbw_color());
                addSmoothFrame(frame);
            }
        } else {
            addSmoothFrame(ledFrame);
        }
    }

    if (pattern.loopFromIndex >= 0 && pattern.frames.size() > 1 &&
        size_t(pattern.loopFromIndex) != pattern.frames.size() - 1 && smooth_ != 0)
    {
        Y_VERIFY(firstLoopFrameDelayMs_ >= 0);
        LedFrame firstLoopFrame = pattern.frames[pattern.loopFromIndex];
        firstLoopFrame.delayMs = uint32_t(firstLoopFrameDelayMs_);

        addSmoothFrame(firstLoopFrame);

        pattern.loopFromIndex++;
    }

    if (gammaCorrection_)
    {
        for (LedFrame& frame : frames)
        {
            frame = applyGammaCorrection(frame);
        }
    }
}

LedFrame LedPattern::getFrameFromStream(std::stringstream& stream, int ledCount)
{
    auto controller = ledDevice_.lock();
    if (!controller) {
        throw std::runtime_error("No controller");
    }

    std::string word;
    stream >> word;
    boost::trim_if(word, boost::is_any_of(" \t"));

    LedFrame ledFrame;
    if (boost::to_lower_copy(word) == "gradient")
    {
        int led1;
        int led2;
        std::string color1;
        std::string color2;
        uint32_t delayMs;
        stream >> led1 >> color1 >> std::dec >> led2 >> color2 >> std::dec >> delayMs;
        return getGradientFrame(led1, controller->readColor(color1), led2, controller->readColor(color2), delayMs, ledCount);
    } else if (ledCount != 0)
    {
        ledFrame.circle.push_back(controller->readColor(word));
    } else if (0 == ledCount)
    {
        std::stringstream tmp;
        tmp << word;
        tmp >> ledFrame.delayMs;
        return ledFrame;
    }

    while ((int)ledFrame.circle.size() < ledCount && stream >> word)
    {
        rgbw_color color = controller->readColor(word);

        ledFrame.circle.push_back(color);
    }

    bool lastValueRead = false;
    uint32_t number;
    while (stream >> std::dec >> number)
    {
        if (!lastValueRead)
        {
            ledFrame.delayMs = number;
            lastValueRead = true;
        } else {
            throw std::runtime_error("Too many tokens in line");
        }
    }

    if (!lastValueRead)
    {
        throw std::runtime_error("Bad LED pattern format");
    }

    return ledFrame;
}

// Do not use pointer or reference for 'frame' because it can be changed by pushing back new frames to pattern
void LedPattern::addSmoothFrame(LedFrame frame)
{
    bool cycleStart = false;
    if (loopFromIndex == int(frames.size())) {
        cycleStart = true;
    }

    if (logSmooth_) {
        addLogSmoothFrame(frame);
    } else {
        addLinearSmoothFrame(frame);
    }

    Y_VERIFY(!frames.empty());

    if (cycleStart) {
        loopFromIndex = int(frames.size()) - 1;
    }
}

// Do not use pointer or reference for 'frame' because it can be changed by pushing back new frames to pattern
void LedPattern::addLinearSmoothFrame(LedFrame frame)
{
    if (frames.empty())
    {
        addFrame(frame);
        return;
    }

    auto lastFrame = frames.back();
    const int lastRotation = int(lastFrame.rotation);
    if (lastFrame.circle == frame.circle)
    {
        addFrame(frame);
        return;
    }

    const uint32_t delayMs = frame.delayMs / (smooth_ + 1);

    for (int i = 0; i < smooth_ + 1; ++i)
    {
        const double frac = double(i + 1) / (smooth_ + 1);

        LedFrame newFrame;
        newFrame.circle.resize(size_t(ledCount_));
        newFrame.delayMs = delayMs;
        for (int j = 0; j < ledCount_; ++j)
        {
            newFrame.circle[j].r = uint8_t(lastFrame.circle[j].r + (frame.circle[j].r - lastFrame.circle[j].r) * frac);
            newFrame.circle[j].g = uint8_t(lastFrame.circle[j].g + (frame.circle[j].g - lastFrame.circle[j].g) * frac);
            newFrame.circle[j].b = uint8_t(lastFrame.circle[j].b + (frame.circle[j].b - lastFrame.circle[j].b) * frac);
            newFrame.circle[j].w = uint8_t(lastFrame.circle[j].w + (frame.circle[j].w - lastFrame.circle[j].w) * frac);
        }

        double nextRotation = frame.rotation;
        if (nextRotation < lastRotation) {
            nextRotation += ledCount_;
        }
        newFrame.rotation = lastRotation + (nextRotation - lastRotation) * frac;

        addFrame(newFrame);
    }
}

// Do not use pointer or reference for 'frame' because it can be changed by pushing back new frames to pattern
void LedPattern::addLogSmoothFrame(LedFrame frame)
{
    if (frames.empty())
    {
        addFrame(frame);
        return;
    }

    auto lastFrame = frames.back();
    if (lastFrame.circle == frame.circle)
    {
        addFrame(frame);
        return;
    }

    const uint32_t delayMs = frame.delayMs / (smooth_ + 1);

    for (int i = 0; i < smooth_ + 1; ++i)
    {
        const double frac = double(i + 1) / (smooth_ + 1);

        LedFrame newFrame;
        newFrame.circle.resize(size_t(ledCount_));
        newFrame.delayMs = delayMs;
        for (int j = 0; j < ledCount_; ++j)
        {
            const double logFromR = log10(lastFrame.circle[j].r + 1);
            const double logToR = log10(frame.circle[j].r + 1);
            const double logFromG = log10(lastFrame.circle[j].g + 1);
            const double logToG = log10(frame.circle[j].g + 1);
            const double logFromB = log10(lastFrame.circle[j].b + 1);
            const double logToB = log10(frame.circle[j].b);
            const double logFromW = log10(lastFrame.circle[j].w + 1);
            const double logToW = log10(frame.circle[j].w);

            newFrame.circle[j].r = uint8_t(std::max(pow(10, logFromR + (logToR - logFromR) * frac) - 1, 0.0));
            newFrame.circle[j].g = uint8_t(std::max(pow(10, logFromG + (logToG - logFromG) * frac) - 1, 0.0));
            newFrame.circle[j].b = uint8_t(std::max(pow(10, logFromB + (logToB - logFromB) * frac) - 1, 0.0));
            newFrame.circle[j].w = uint8_t(std::max(pow(10, logFromW + (logToW - logFromW) * frac) - 1, 0.0));
        }

        addFrame(newFrame);
    }
}

void LedPattern::addFrame(LedFrame frame)
{
    if (optimizeSize_ && !frames.empty() && frame.circle == frames.back().circle) {
        frames.back().delayMs += frame.delayMs;
    } else {
        frames.push_back(frame);
    }
}

LedFrame LedPattern::getRotatedFrame(const LedFrame& original, int count, int ledCount)
{
    LedFrame rotated;
    rotated.circle.resize(size_t(ledCount));
    rotated.delayMs = original.delayMs;
    if (count < 0) {
        count += ledCount;
    }

    std::copy(original.circle.begin() + ledCount - count, original.circle.end(), rotated.circle.begin());
    std::copy(original.circle.begin(), original.circle.begin() + ledCount - count, rotated.circle.begin() + count);

    rotated.rotation = count;

    return rotated;
}

LedFrame LedPattern::getRotatedFrame(const LedFrame& original, double count, int ledCount)
{
    const double eps = 1e-8;
    int rotationBefore = int(floor(count + eps));
    LedFrame before = getRotatedFrame(original, rotationBefore, ledCount);
    if (rotationBefore >= ledCount) {
        return before;
    }

    LedFrame after = getRotatedFrame(original, rotationBefore + 1, ledCount);
    const double frac = count - rotationBefore;

    for (int i = 0; i < ledCount; ++i)
    {
        before.circle[i].r += (after.circle[i].r - before.circle[i].r) * frac;
        before.circle[i].g += (after.circle[i].g - before.circle[i].g) * frac;
        before.circle[i].b += (after.circle[i].b - before.circle[i].b) * frac;
        before.circle[i].w += (after.circle[i].w - before.circle[i].w) * frac;
    }

    before.rotation = count;

    return before;
}

LedFrame LedPattern::getGradientFrame(int led1, const rgbw_color& color1, int led2, const rgbw_color& color2,
                                      uint32_t delayMs, int ledCount)
{
    if (led1 == led2)
    {
        throw std::runtime_error("Cannot create gradient between two same points: led1 = " + std::to_string(led1) + " led2 = " + std::to_string(led2));
    }

    if (led1 < 0 || led2 < 0 || led1 >= ledCount || led2 >= ledCount)
    {
        throw std::runtime_error("Cannot create gradient between two points: led1 = " + std::to_string(led1) + " led2 = " + std::to_string(led2) + " ledCount = " + std::to_string(ledCount));
    }

    LedFrame result;
    result.delayMs = delayMs;
    result.circle.resize(size_t(ledCount));

    int from = led1;
    int to = led2;
    rgbw_color colorFrom = color1;
    rgbw_color colorTo = color2;

    if (from > to)
    {
        from = led2;
        to = led1;
        colorFrom = color2;
        colorTo = color1;
    }

    const double eps = 0.0000000001;

    for (int i = from; i <= to; ++i)
    {
        const double percent = (i - from) / double(to - from);
        result.circle[i].r = uint8_t(round(colorFrom.r + (colorTo.r - colorFrom.r) * percent) + eps);
        result.circle[i].g = uint8_t(round(colorFrom.g + (colorTo.g - colorFrom.g) * percent) + eps);
        result.circle[i].b = uint8_t(round(colorFrom.b + (colorTo.b - colorFrom.b) * percent) + eps);
        result.circle[i].w = uint8_t(round(colorFrom.w + (colorTo.w - colorFrom.w) * percent) + eps);
    }

    for (int i = (to + 1) % ledCount; i != from; i = (i + 1) % ledCount)
    {
        int distance = (ledCount - i) + from;
        const double percent = distance / double(ledCount - (to - from));
        result.circle[i].r = uint8_t(round(colorFrom.r + (colorTo.r - colorFrom.r) * percent) + eps);
        result.circle[i].g = uint8_t(round(colorFrom.g + (colorTo.g - colorFrom.g) * percent) + eps);
        result.circle[i].b = uint8_t(round(colorFrom.b + (colorTo.b - colorFrom.b) * percent) + eps);
        result.circle[i].w = uint8_t(round(colorFrom.w + (colorTo.w - colorFrom.w) * percent) + eps);
    }

    return result;
}

std::vector<LedFrame> LedPattern::getFrames() const {
    return frames;
}

int LedPattern::getLedCount() const {
    return ledCount_;
}

bool LedPattern::isBackground() const {
    return isBackground_;
}

std::string LedPattern::getName() const {
    return name_;
}

LedFrame LedPattern::applyGammaCorrection(const LedFrame& original)
{
    // clang-format off
    const uint8_t gamma8[] = {
        0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
        1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
        1,  1,  1,  1,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  3,  3,
        3,  3,  3,  3,  3,  4,  4,  4,  4,  4,  5,  5,  5,  5,  5,  6,
        6,  6,  6,  7,  7,  7,  7,  8,  8,  8,  8,  9,  9,  9, 10, 10,
       10, 11, 11, 12, 12, 12, 13, 13, 13, 14, 14, 15, 15, 16, 16, 17,
       17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 24, 24, 25, 25,
       26, 27, 27, 28, 29, 29, 30, 31, 31, 32, 33, 34, 34, 35, 36, 37,
       38, 38, 39, 40, 41, 42, 43, 43, 44, 45, 46, 47, 48, 49, 50, 51,
       52, 53, 54, 55, 56, 57, 58, 59, 60, 62, 63, 64, 65, 66, 67, 68,
       70, 71, 72, 73, 75, 76, 77, 78, 80, 81, 82, 84, 85, 87, 88, 89,
       91, 92, 94, 95, 97, 98,100,101,103,104,106,108,109,111,112,114,
      116,117,119,121,123,124,126,128,130,131,133,135,137,139,141,143,
      145,147,149,151,153,155,157,159,161,163,165,167,169,171,173,176,
      178,180,182,185,187,189,192,194,196,199,201,203,206,208,211,213,
      216,218,221,223,226,228,231,234,236,239,242,244,247,250,253,255,
    };
    // clang-format on

    LedFrame result = original;

    for (auto& color : result.circle)
    {
        color.r = gamma8[color.r];
        color.g = gamma8[color.g];
        color.b = gamma8[color.b];
        color.w = gamma8[color.w];
    }

    return result;
}

void LedPattern::setSmooth(int smooth)
{
    smooth_ = smooth;
}

bool LedPattern::empty() const {
    return frames.empty();
}

std::shared_ptr<LedPattern> LedPattern::rotateByAngle(double angle)
{
    auto rotated = std::make_shared<LedPattern>(*this);
    rotated->frames.clear();

    for (const LedFrame& frame : frames)
    {
        rotated->frames.push_back(getRotatedFrame(frame, (angle / 360.) * ledCount_, ledCount_));
    }

    return rotated;
}

std::shared_ptr<LedPattern> LedPattern::getIdlePattern(int ledCount,
                                                       const std::weak_ptr<quasar::LedController>& ledController)
{
    auto result = std::make_shared<LedPattern>(ledCount, ledController);
    result->name_ = "idle.led";
    result->frames.push_back(getIdleFrame(300000, ledCount));
    result->loopFromIndex = 0;

    return result;
}

LedFrame LedPattern::getIdleFrame(int delayMs, int ledCount)
{
    LedFrame result;
    result.circle.resize(size_t(ledCount), rgbw_color());
    result.delayMs = uint32_t(delayMs);
    return result;
}

std::string LedPattern::getInRawFormat() const {
    std::stringstream str;
    str << std::setfill('0') << std::hex << std::setw(2);
    for (size_t i = 0; i < frames.size(); ++i)
    {
        if (loopFromIndex == int(i)) {
            str << "loop" << std::endl;
        }

        for (size_t j = 0; j < frames[i].circle.size(); ++j)
        {
            str << std::hex << std::setw(2) << uint32_t(frames[i].circle[j].r)
                << std::hex << std::setw(2) << uint32_t(frames[i].circle[j].g)
                << std::hex << std::setw(2) << uint32_t(frames[i].circle[j].b)
                << std::hex << std::setw(2) << uint32_t(frames[i].circle[j].w)
                << " ";
        }

        str << std::dec << frames[i].delayMs << "\n";
    }

    return str.str();
}

bool LedPattern::finished() const {
    return (currentFrame_ >= (int)frames.size()) || (std::chrono::steady_clock::now() >= endOfDuration_);
}

bool LedPattern::started() const {
    return started_;
}

LedFrame LedPattern::getCurrentFrame() const {
    Y_VERIFY(currentFrame_ < (int)frames.size());
    return frames[currentFrame_];
}

LedPattern::TimePoint LedPattern::getEndOfFrameTimePoint() const {
    return std::min(endOfFrame_, endOfDuration_);
}

void LedPattern::setDuration(int durationMs)
{
    setEnd(std::chrono::steady_clock::now() + std::chrono::milliseconds(durationMs));
}

void LedPattern::setEnd(TimePoint endOfDuration)
{
    endOfDuration_ = endOfDuration;
}

void LedPattern::updateTime(Animation::TimePoint timePoint) {
    if (!started_) {
        return;
    }
    auto delta = timePoint - startOfCurrentFrame_;
    while (!finished() && (delta >= std::chrono::milliseconds(getCurrentFrame().delayMs))) {
        auto previousFrameDelay = std::chrono::milliseconds(getCurrentFrame().delayMs);
        startOfCurrentFrame_ += previousFrameDelay;
        delta -= previousFrameDelay;
        currentFrame_ += 1;
        if (!finished()) {
            endOfFrame_ = startOfCurrentFrame_ + std::chrono::milliseconds(getCurrentFrame().delayMs);
        }
    }

    if (currentFrame_ >= (int)frames.size() && loopFromIndex >= 0) {
        currentFrame_ = loopFromIndex;
        startOfCurrentFrame_ = timePoint;
        endOfFrame_ = startOfCurrentFrame_ + std::chrono::milliseconds(getCurrentFrame().delayMs);
    }
}

void LedPattern::startAnimationFrom(Animation::TimePoint startTime) {
    if (started_) {
        return;
    }
    started_ = true;
    startTime_ = startTime;
    startOfCurrentFrame_ = startTime;
    currentFrame_ = 0;
    // endOfDuration is handled separately
    if (!finished()) {
        endOfFrame_ = startTime + std::chrono::milliseconds(getCurrentFrame().delayMs);
    }
}

void LedPattern::drawCurrentFrame() {
    if (!started_) {
        return;
    }
    LedFrame frame = getCurrentFrame();
    if (auto ledController = ledDevice_.lock()) {
        ledController->drawFrame(frame.circle);
    }
}

void LedPattern::resetAnimation() {
    started_ = false;
    currentFrame_ = 0;
}

std::chrono::nanoseconds LedPattern::getLength() const {
    auto length = std::chrono::nanoseconds(0);
    for (const auto& frame : frames) {
        length += std::chrono::milliseconds(frame.delayMs);
    }
    return length;
}
