#pragma once

#include <yandex_io/modules/leds/led_controller/color_readers.h>
#include <yandex_io/modules/leds/led_controller/led_circle.h>
#include <yandex_io/modules/leds/led_manager/ng/animation.h>

#include <cstdint>
#include <functional>
#include <memory>
#include <string>
#include <vector>

namespace quasar {

    struct LedFrame {
        LedCircle circle;
        uint32_t delayMs = 0;
        double rotation = 0;
    };

    class LedPattern: public Animation {
    public:
        LedPattern(int ledCount, std::weak_ptr<quasar::LedController> ledController);
        LedPattern(int ledCount, std::string name, std::weak_ptr<quasar::LedController> ledController);
        LedPattern(const LedPattern& other) = default;

        static std::shared_ptr<LedPattern> getIdlePattern(int ledCount, const std::weak_ptr<quasar::LedController>& ledController);
        static LedFrame getRotatedFrame(const LedFrame& original, int count, int ledCount);
        static LedFrame getRotatedFrame(const LedFrame& original, double count, int ledCount);
        static LedFrame getIdleFrame(int delayMs, int ledCount);
        std::weak_ptr<quasar::LedController> getDevice();

        void load(std::istream& stream);

        void setSmooth(int smooth);

        void addSmoothFrame(LedFrame frame);

        static std::shared_ptr<LedPattern> loadFromFile(const std::string& fileName, const std::weak_ptr<quasar::LedController>& ledController);

        virtual ~LedPattern() = default;

        bool empty() const;

        std::shared_ptr<LedPattern> rotateByAngle(double angle); // TODO: make rotatedByAngle. Also maybe review signature to make it clear copy

        std::string getInRawFormat() const;

        std::vector<LedFrame> getFrames() const;

        int getLedCount() const;

        bool isBackground() const;

        std::string getName() const override;

        bool finished() const override;

        bool started() const;

        virtual LedFrame getCurrentFrame() const;

        TimePoint getEndOfFrameTimePoint() const override;

        void setDuration(int durationMs);

        void drawCurrentFrame() override;

        void resetAnimation() override;
        void updateTime(TimePoint timePoint) override;
        void startAnimationFrom(TimePoint startTime) override;
        std::chrono::nanoseconds getLength() const override;

        int loopFromIndex = -1;
        std::vector<LedFrame> frames;

    protected:
        int ledCount_;
        bool isBackground_ = false;

        int currentFrame_ = 0;
        TimePoint endOfFrame_;
        TimePoint endOfDuration_ = std::chrono::steady_clock::time_point::max();
        bool started_ = false;
        bool optimizeSize_ = true;
        TimePoint startOfCurrentFrame_;

    private:
        void addLinearSmoothFrame(LedFrame frame);
        void addLogSmoothFrame(LedFrame frame);

        void addFrame(LedFrame frame);

        LedFrame getFrameFromStream(std::stringstream& stream, int ledCount);

        static LedFrame getGradientFrame(int led1, const rgbw_color& color1, int led2, const rgbw_color& color2,
                                         uint32_t delayMs, int ledCount);

        static LedFrame applyGammaCorrection(const LedFrame& original);

        int smooth_ = 0;
        bool logSmooth_ = false;
        bool gammaCorrection_ = false;
        int firstLoopFrameDelayMs_ = -1;

        std::string name_;
        std::weak_ptr<quasar::LedController> ledDevice_;
        // TODO: remove?
        TimePoint startTime_;
        void setEnd(TimePoint endOfDuration);
    };

} // namespace quasar
