#include "led_pattern_storage.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/generic/scope.h>

#include <dirent.h>

YIO_DEFINE_LOG_MODULE("leds");

using namespace quasar;

LedPatternStorage::LedPatternStorage(const std::string& path, std::shared_ptr<quasar::LedController> controller)
    : controller_(std::move(controller))
{
    DIR* dp = opendir(path.c_str());

    if (dp == nullptr) {
        YIO_LOG_ERROR_EVENT("LedPatternStorage.OpenDirFailed", "Cannot open directory " + path);
        throw std::runtime_error("Cannot open directory " + path);
    }

    Y_DEFER {
        closedir(dp);
    };

    dirent* ep = nullptr;
    while ((ep = readdir(dp)) != nullptr) {
        std::string file = ep->d_name;
        if (file.ends_with(".led")) {
            std::string pathToPattern = path;
            if (pathToPattern.back() != '/') {
                pathToPattern += '/';
            }
            pathToPattern += file;
            try {
                patterns_[file] = LedPattern::loadFromFile(pathToPattern, controller_);
            } catch (const std::exception& e) {
                YIO_LOG_WARN("Cannot load pattern from file " << pathToPattern << ": " << e.what());
            }
        }
    }
}

std::shared_ptr<LedPattern> LedPatternStorage::getPattern(const std::string& name) const {
    auto pattern = std::make_shared<LedPattern>(*patterns_.at(name));
    pattern->resetAnimation();
    return pattern;
}

std::shared_ptr<ListeningPattern> LedPatternStorage::getListeningPattern(double doaAngle) const {
    const bool invertRotation = false;
    return std::make_shared<ListeningPattern>(getPattern("alice_listening.led"), doaAngle, invertRotation);
}

std::shared_ptr<VolumePattern> LedPatternStorage::getVolumePattern(double volume, bool looped) const {
    auto baseVolumePattern = getPattern("volume.led");
    return std::make_shared<VolumePattern>(std::move(baseVolumePattern), volume, looped);
}

std::shared_ptr<ProgressPattern> LedPatternStorage::getProgressPattern() const {
    return std::make_shared<ProgressPattern>(getPattern("progress.led"));
}

std::shared_ptr<LedPattern> LedPatternStorage::getIdlePattern() const {
    return LedPattern::getIdlePattern(controller_->getLedCount(), controller_);
}

const std::shared_ptr<quasar::LedController>& LedPatternStorage::getController() const {
    return controller_;
}
