#include "listening_pattern.h"

#include <yandex_io/libs/logging/logging.h>

#include <chrono>
#include <cmath>

using namespace quasar;

namespace {
    const double ROTATION_SPEED_DEGREES_PER_SECOND = 150.;
} // namespace

ListeningPattern::ListeningPattern(std::shared_ptr<LedPattern> base, double doaAngle, bool invertRotation)
    : LedPattern(*base)
    , doaAngle_(doaAngle)
    , currentRotationAngle_(doaAngle)
    , invertRotation_(invertRotation)
    , finished_(false)
    , lastRotationUpdate_(TimePoint::clock::now())
    , animationUpdate_(TimePoint::clock::now())
{
    if (loopFromIndex >= 0)
    {
        throw std::runtime_error("loops in listening pattern are not supported");
    }
}

bool ListeningPattern::finished() const {
    return finished_;
}

void ListeningPattern::resetAnimation()
{
    finished_ = false;
    LedPattern::resetAnimation();
}

void ListeningPattern::finish() {
    finished_ = true;
}

void ListeningPattern::startAnimationFrom(Animation::TimePoint startTime) {
    if (!started()) {
        startOfCurrentFrame_ = startTime;
        started_ = true;
        currentFrame_ = 0;
        lastRotationUpdate_ = startTime;
    }

    if (currentFrame_ < static_cast<int>(frames.size())) {
        endOfFrame_ = startTime + std::chrono::milliseconds(frames[currentFrame_].delayMs);
    }
}

void ListeningPattern::updateTime(TimePoint timePoint) {
    animationUpdate_ = timePoint;
    LedPattern::updateTime(timePoint);
}

LedFrame ListeningPattern::getCurrentFrame() const {
    const double doaAngle = doaAngle_;
    const auto absAngleDiff = std::fabs(doaAngle - currentRotationAngle_);
    if (absAngleDiff > 1.) {
        using FloatSeconds = std::chrono::duration<double>;
        const auto timeDiff = std::chrono::duration_cast<FloatSeconds>(animationUpdate_ - lastRotationUpdate_);
        const double absStep = ROTATION_SPEED_DEGREES_PER_SECOND * timeDiff.count();
        if (absStep < absAngleDiff && absStep < 360. / ledCount_) {
            const double step = std::copysign(absStep, (doaAngle - currentRotationAngle_) * ((360 - absAngleDiff) - absAngleDiff));
            currentRotationAngle_ += step;
            if (currentRotationAngle_ < 0) {
                currentRotationAngle_ += 360;
            } else if (currentRotationAngle_ > 359) {
                currentRotationAngle_ -= 360;
            }
        }
    }
    const double angle = (invertRotation_) ? 360. - currentRotationAngle_ : currentRotationAngle_;
    const auto framesSize = static_cast<int>(frames.size());
    lastRotationUpdate_ = animationUpdate_;
    return getRotatedFrame(frames[std::min(currentFrame_, framesSize - 1)], (angle / 360.) * ledCount_, ledCount_);
}

void ListeningPattern::setDoaAngle(double doaAngle)
{
    doaAngle_ = doaAngle;
    if (finished_) {
        currentRotationAngle_ = doaAngle;
    }
}

double ListeningPattern::getDoaAngle() const {
    return doaAngle_;
}
