#include "device_power_on_metrica.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <util/system/yassert.h>

#include <chrono>
#include <fstream>

namespace {

    std::optional<int32_t> readLastMonotonicTimestamp(const std::string& file) {
        Y_VERIFY(!file.empty());
        if (!quasar::fileExists(file)) {
            return std::nullopt;
        }
        std::ifstream f(file);

        int32_t val = -1;
        if (f.good() && f >> val) {
            return val;
        }

        return std::nullopt;
    }

    void sendMetrica(YandexIO::ITelemetry& telemetry, std::optional<int32_t> powerOffTs) {
        const auto now = std::chrono::system_clock::now();
        const int32_t nowTs = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();

        Json::Value payload;
        payload["powerOffTimestampSec"] = powerOffTs.value_or(-1);
        payload["nowTimestampSec"] = nowTs;
        telemetry.reportEvent("device_power_on", quasar::jsonToString(payload));
    }

} // namespace

namespace YandexIO {

    PowerOnMetricaSender::PowerOnMetricaSender(std::shared_ptr<ITelemetry> telemetry,
                                               std::string guardFile,
                                               const std::string& monotonicTimeFile)
        : powerOffTs_(readLastMonotonicTimestamp(monotonicTimeFile))
        , guardFile_(std::move(guardFile))
        , telemetry_(std::move(telemetry))
        , sentOncePerBoot_(quasar::fileExists(guardFile_))
    {
        Y_VERIFY(telemetry_);
        Y_VERIFY(!guardFile_.empty());
    }

    void PowerOnMetricaSender::onSDKState(const SDKState& state) {
        if (sentOncePerBoot_) {
            return;
        }
        if (!state.ntpdState.clockSynchronized.value_or(false)) {
            return;
        }
        sendMetrica(*telemetry_, powerOffTs_);

        sentOncePerBoot_ = true;
        std::ofstream f(guardFile_);
    }

} // namespace YandexIO
