#include "bio_capability.h"

#include "enrollment_engine_safety_wrapper.h"
#include "sound_setup/bio_sound_setup.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/cryptography/cryptography.h>
#include <yandex_io/libs/hal/hal.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <yandex_io/protos/account_storage.pb.h>

#include <json/config.h>
#include <json/value.h>

#include <library/cpp/json/json_writer.h>

#include <util/system/yassert.h>

#include <cstdint>

#include <algorithm>
#include <vector>

YIO_DEFINE_LOG_MODULE("BioCapability");

namespace {

    constexpr auto DEFAULT_ENROLLMENT_TIMEOUT = std::chrono::minutes{10};

    NAlice::EUserType userTypeToProto(quasar::UserType userType) {
        switch (userType) {
            case quasar::UserType::OWNER:
                return NAlice::EUserType::OWNER;
            case quasar::UserType::GUEST:
                return NAlice::EUserType::GUEST;
        };
    }

    std::string userTypeToString(quasar::UserType userType) {
        return PROTO_ENUM_TO_STRING(NAlice::EUserType, userTypeToProto(userType));
    }

    std::optional<quasar::UserType> userTypeFromString(std::string userType) {
        if (userType == "OWNER") {
            return quasar::UserType::OWNER;
        } else if (userType == "GUEST") {
            return quasar::UserType::GUEST;
        }

        return std::nullopt;
    }

    NAlice::TCapabilityHolder createCapabilityState() {
        NAlice::TCapabilityHolder capabilityHolder;
        auto bioCapability = capabilityHolder.mutable_biocapability();

        bioCapability->mutable_meta();
        bioCapability->mutable_parameters();
        bioCapability->mutable_state();

        return capabilityHolder;
    }

} // namespace

using namespace quasar;
using namespace YandexIO;

BioCapability::RequestListener::RequestListener(BioCapability& bioCapability)
    : bioCapability_{bioCapability}
    , isRequestActive_{false}
{
}

void BioCapability::RequestListener::startRequest()
{
    isRequestActive_.store(true);
}

void BioCapability::RequestListener::finishRequest()
{
    isRequestActive_.store(false);
}

void BioCapability::RequestListener::onAudioData(const ChannelsData& channels)
{
    if (!isRequestActive_.load()) {
        return;
    }

    auto mainChannelIt = std::find_if(channels.begin(), channels.end(), [](const auto& channel) { return channel.isForRecognition; });
    Y_VERIFY(mainChannelIt != channels.end());
    bioCapability_.onSoundBuffer(*mainChannelIt);
}

std::unique_ptr<BioCapability> BioCapability::create(
    std::shared_ptr<IDevice> device,
    std::shared_ptr<SDKInterface> sdk,
    std::shared_ptr<quasar::IAuthProvider> authProvider,
    std::shared_ptr<IAudioSourceClient> audioSourceClient,
    BioConfig bioConfig)
{
    auto enrollmentEngine = std::make_shared<EnrollmentEngineSafetyWrapper>(bioConfig.modelPath);

    return std::make_unique<BioCapability>(
        std::move(device),
        std::move(sdk),
        std::move(authProvider),
        std::move(audioSourceClient),
        std::move(enrollmentEngine),
        std::move(bioConfig),
        std::make_shared<NamedCallbackQueue>("BioCapability"));
}

// Constructor for tests
BioCapability::BioCapability(std::shared_ptr<IDevice> device,
                             std::shared_ptr<SDKInterface> sdk,
                             std::shared_ptr<quasar::IAuthProvider> authProvider,
                             std::shared_ptr<IAudioSourceClient> audioSource,
                             std::shared_ptr<NBio::NDevice::IEnrollmentEngine> enrollmentEngine,
                             BioConfig config,
                             std::shared_ptr<quasar::ICallbackQueue> asyncQueue)
    : device_{std::move(device)}
    , sdk_{std::move(sdk)}
    , enrollmentEngine_{std::move(enrollmentEngine)}
    , authProvider_{std::move(authProvider)}
    , audioSource_{std::move(audioSource)}
    , bioConfig_{std::move(config)}
    , capabilityState_{createCapabilityState()}
    , asyncQueue_{std::move(asyncQueue)}
    , cancelEnrollmentCallback_{asyncQueue_}
    , addGuestTimeoutCallback_{asyncQueue_}
    , soundEnrollmentTimeoutCallback_{asyncQueue_}
{
    Y_VERIFY(device_);
    Y_VERIFY(audioSource_);
    Y_VERIFY(authProvider_);
    Y_VERIFY(enrollmentEngine_);
    Y_VERIFY(asyncQueue_);

    initSupportedDirectiveNames();

    asyncQueue_->add([this] {
        init();
    }, lifetime_);
}

void BioCapability::setBioSoundSetup(std::unique_ptr<BioSoundSetup> soundSetup)
{
    soundSetup_ = std::move(soundSetup);
}

BioCapability::~BioCapability()
{
    lifetime_.die();
}

NAlice::TCapabilityHolder BioCapability::getState() const {
    return capabilityState_;
}

YandexIO::IDirectiveHandlerPtr BioCapability::getDirectiveHandler()
{
    return shared_from_this();
}

void BioCapability::addListener(std::weak_ptr<ICapability::IListener> wlistener) {
    Y_UNUSED(wlistener);
}

void BioCapability::removeListener(std::weak_ptr<ICapability::IListener> wlistener) {
    Y_UNUSED(wlistener);
}

const std::string& BioCapability::getEndpointId() const {
    return device_->deviceId();
}

const std::string& BioCapability::getHandlerName() const {
    static const std::string CAPABILITY_NAME{"BioCapability"};
    return CAPABILITY_NAME;
}

const std::set<std::string>& BioCapability::getSupportedDirectiveNames() const {
    return supportedDirectiveNames_;
}

void BioCapability::initSupportedDirectiveNames() {
    supportedDirectiveNames_ = {
        Directives::UPDATE_VOICE_PRINTS,
        Directives::MATCH_VOICE_PRINT,
        Directives::BIO_REMOVE_ACCOUNT,
        Directives::BIO_REMOVE_ACCOUNT2,
        Directives::ENROLLMENT_START,
        Directives::ENROLLMENT_CANCEL,
        Directives::ENROLLMENT_FINISH,
    };

    if (bioConfig_.soundSetup) {
        supportedDirectiveNames_.insert(Directives::BIO_START_SOUND_ENROLLMENT);
    } else {
        supportedDirectiveNames_.insert(Directives::BIO_ADD_ACCOUNT);
        supportedDirectiveNames_.insert(Directives::BIO_ADD_ACCOUNT2);
    }
}

void BioCapability::handleDirective(const std::shared_ptr<Directive>& directive)
{
    asyncQueue_->add([this, directive] {
        if (directive->is(Directives::UPDATE_VOICE_PRINTS)) {
            updateVoicePrints(directive->getData().payload);
        } else if (directive->is(Directives::MATCH_VOICE_PRINT)) {
            parentMatchMessageId_ = directive->getData().parentMessageId;
            matchVoicePrint(directive->getData().payload);
        } else if (directive->is(Directives::BIO_ADD_ACCOUNT) || directive->is(Directives::BIO_ADD_ACCOUNT2)) {
            addAccount(directive->getData().payload);
        } else if (directive->is(Directives::BIO_REMOVE_ACCOUNT) || directive->is(Directives::BIO_REMOVE_ACCOUNT2)) {
            removeAccount(directive->getData().payload);
        } else if (directive->is(Directives::ENROLLMENT_START)) {
            startEnrollment(directive->getData().payload);
        } else if (directive->is(Directives::ENROLLMENT_CANCEL)) {
            cancelEnrollment(directive->getData().payload);
        } else if (directive->is(Directives::ENROLLMENT_FINISH)) {
            finishEnrollment(directive->getData().payload);
        } else if (directive->is(Directives::BIO_START_SOUND_ENROLLMENT)) {
            startSoundEnrollment();
        }
    }, lifetime_);
}

void BioCapability::cancelDirective(const std::shared_ptr<Directive>& directive)
{
    Y_UNUSED(directive);
}

void BioCapability::prefetchDirective(const std::shared_ptr<Directive>& directive)
{
    Y_UNUSED(directive);
}

void BioCapability::init()
{
    Y_ENSURE_THREAD(asyncQueue_);

    auto cryptographyConfig = getJson(device_->configuration()->getServiceConfig("common"), "cryptography");
    deviceCryptography_ = device_->hal()->createDeviceCryptography(cryptographyConfig);

    loadFileStorage();

    audioListener_ = std::make_shared<RequestListener>(*this);
    audioSource_->addListener(audioListener_);

    authProvider_->usersAuthInfo().connect([this](const auto& usersAuthInfo) {
        onAccountsChanged(*usersAuthInfo);
    }, lifetime_, asyncQueue_);
}

void BioCapability::onAccountsChanged(const IAuthProvider::UsersAuthInfo& usersAuthInfo)
{
    Y_ENSURE_THREAD(asyncQueue_);

    // It's empty on start because AliceService starts before AuthService. https://a.yandex-team.ru/arc_vcs/yandex_io/scaffolding/yiod/yiod.cpp?rev=c329000933fa39fd7aaa34227046638091fd4dc3#L227
    // So we get default usersAuthInfo (which is empty) https://a.yandex-team.ru/arc_vcs/yandex_io/interfaces/auth/connector/auth_provider.cpp?rev=c329000933fa39fd7aaa34227046638091fd4dc3#L128
    if (usersAuthInfo.empty()) {
        return;
    }
    // Delete old users and enrolls for them
    for (auto it = users_.begin(); it != users_.end();) {
        const auto foundAuthInfo = std::find_if(
            usersAuthInfo.cbegin(), usersAuthInfo.cend(),
            [puid = it->first](const auto& authInfo) {
                return puid == authInfo.passportUid;
            });

        if (foundAuthInfo == usersAuthInfo.cend()) {
            const auto& [_, user] = *it;
            if (!user.enrollId.empty()) {
                enrollments_.erase(user.enrollId);
            }
            it = users_.erase(it);
        } else {
            ++it;
        }
    }

    // Fill user type and token for valid users
    for (const auto& authInfo : usersAuthInfo) {
        users_[authInfo.passportUid].authToken = authInfo.authToken;
        users_[authInfo.passportUid].userType = authInfo.userType;
    }

    updateEnrollments();
    updateFileStorage();
    updateEnrollmentHeaders();
}

void BioCapability::loadFileStorage()
{
    YIO_LOG_INFO("Load users");

    const auto maybeJson = tryReadJsonFromFile(bioConfig_.storagePath);
    if (!maybeJson.has_value()) {
        return;
    }

    const auto& jsonStorage = *maybeJson;
    if (!jsonStorage.isArray()) {
        YIO_LOG_WARN("Wrong storage json, consider storage as empty");
        return;
    }

    for (const auto& node : jsonStorage) {
        auto enrollId = tryGetString(node, "enrollId");
        auto puid = tryGetString(node, "puid");

        YIO_LOG_DEBUG("Load user. puid: '" << puid << "', enrollId: '" << enrollId << "'");

        if (!enrollId.empty()) {
            enrollments_[enrollId] = TBlob::FromString(base64Decode(tryGetString(node, "voiceprint")));
        }

        users_[puid].enrollId = std::move(enrollId);
    }
}

void BioCapability::updateFileStorage() const {
    Y_ENSURE_THREAD(asyncQueue_);

    Json::Value jsonStorage(Json::arrayValue);
    for (const auto& [puid, user] : users_) {
        YIO_LOG_DEBUG("Write user. puid: '" << puid << "', enrollId: '" << user.enrollId << "'");

        Json::Value jsonUser;
        jsonUser["enrollId"] = user.enrollId;
        jsonUser["puid"] = puid;
        if (auto enroll = enrollments_.find(user.enrollId); enroll != enrollments_.cend()) {
            const auto& voiceprint = enroll->second;
            jsonUser["voiceprint"] = base64Encode(voiceprint.AsCharPtr(), voiceprint.Size());
        }
        jsonStorage.append(std::move(jsonUser));
    }

    YIO_LOG_INFO("Write storage " << jsonStorage.size() << " users");

    bool writeSuccess = true;
    try {
        PersistentFile storage(bioConfig_.storagePath, PersistentFile::Mode::TRUNCATE);
        writeSuccess = storage.write(jsonToString(jsonStorage));
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("BioCapability.WriteStorageFailed", "Couldn't write users to storage. " << ex.what());
    }

    if (!writeSuccess) {
        YIO_LOG_ERROR_EVENT("BioCapability.WriteStorageFailed", "Couldn't write users to storage");
    }
}

void BioCapability::updateEnrollments()
{
    YIO_LOG_INFO("Set enrollments " << enrollments_.size());
    enrollmentEngine_->SetEnrollments(enrollments_);
}

void BioCapability::addAccount(const Json::Value& payload)
{
    if (bioConfig_.soundSetup) {
        const std::string errorMessage{"Add account directive isn't supported"};
        YIO_LOG_ERROR_EVENT("BioCapability.UnsupportedDirective", errorMessage);
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
    }

    device_->telemetry()->reportEvent("bioAddAccount");

    if (enrollmentEngine_->IsEnrollmentActive()) {
        const std::string errorMessage = "Another enrollment is active";
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage);
        reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
        return;
    }

    const auto userTypeString = tryGetString(payload, "user_type", "GUEST");
    const auto userType = userTypeFromString(userTypeString);
    if (!userType.has_value()) {
        const std::string errorMessage = "Invalid user type: " + userTypeString;
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage);
        reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
        return;
    }

    if (userType == UserType::OWNER) {
        addOwnerAccount();
    } else {
        addGuestAccount(payload);
    }
}

void BioCapability::addGuestAccount(const Json::Value& payload) {
    Y_ENSURE_THREAD(asyncQueue_);

    const auto encryptedSessionKey = base64Decode(tryGetString(payload, "encrypted_session_key"));
    const auto encryptedCode = base64Decode(tryGetString(payload, "encrypted_code"));
    const auto signature = base64Decode(tryGetString(payload, "signature"));
    const auto tokenType = tryGetString(payload, "token_type");

    YIO_LOG_DEBUG("encryptedSessionKey length: " << encryptedSessionKey.length());
    YIO_LOG_DEBUG("encryptedCode length: " << encryptedCode.length());
    YIO_LOG_DEBUG("signature length: " << signature.length());
    YIO_LOG_DEBUG("tokenType: " << tokenType);

    if (tokenType != "XToken" && tokenType != "OAuthToken") {
        const std::string errorMessage = "Invalid token type: " + tokenType;
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage);
        reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
        return;
    }

    const bool withXToken = tokenType == "XToken";

    const auto sessionKey = deviceCryptography_->decrypt(encryptedSessionKey);

    YIO_LOG_DEBUG("sessionKey length: " << sessionKey.length());

    Cryptography cryptography;
    if (const auto hmac = Cryptography::hashWithHMAC_SHA256(encryptedCode, sessionKey); signature != hmac) {
        const std::string errorMessage = "Can't decrypt code. Wrong signature: " + hmac;
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage);
        reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
        return;
    }

    const std::string_view IV(sessionKey.data(), 16);
    const std::string_view key(sessionKey.data() + 16, sessionKey.size() - 16);

    YIO_LOG_DEBUG("IV length: " << IV.length());
    YIO_LOG_DEBUG("key length: " << key.length());

    const std::string decryptedCode = Cryptography::decryptAES(IV, key, encryptedCode);
    std::vector<unsigned char> decryptedBytes(decryptedCode.begin(), decryptedCode.end());
    const auto code = bytesToString(decryptedBytes);

    YIO_LOG_DEBUG("code length: " << code.size());

    addGuestAccount(code, withXToken);
}

void BioCapability::addGuestAccount(const std::string& authCode, bool withXToken) {
    YIO_LOG_INFO("Add guest account");

    Y_ENSURE_THREAD(asyncQueue_);

    const std::chrono::seconds timeout{10};
    const auto addUserResult = authProvider_->addUser(authCode, UserType::GUEST, withXToken, timeout);
    if (addUserResult.status != IAuthProvider::AddUserResponse::Status::OK) {
        const std::string errorMessage = "AddUser failed: " + addUserResult.statusName();
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage)
        reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
        sendEnrollmentStatusSemanticFrame(false, "PassportError", "Couldn't get passport credentials");
        onEnrollmentError();

        // Add more attempts to parse sound
        if (soundSetup_ != nullptr) {
            soundSetup_->startParsing(soundSetupTimeout_);
        }

        return;
    }

    if (soundSetup_) {
        soundSetup_->stopParsing();
    }

    soundEnrollmentTimeoutCallback_.reset();

    enrollmentPuid_ = std::to_string(addUserResult.id);

    reportMetricaEvent(MetricaEvent{.name = "bioAddAccountSuccess", .puid = *enrollmentPuid_});
    sendGuestEnrollmentStartSemanticFrame(*enrollmentPuid_, addUserResult.authToken);

    const std::chrono::minutes addGuestTimeout{10};
    addGuestTimeoutCallback_.executeDelayed([this, timeout = addGuestTimeout, puid = *enrollmentPuid_, token = addUserResult.authToken] {
        const std::string errorMessage{"Add account timeout"};
        YIO_LOG_ERROR_EVENT("BioCapability.AddAccountTimeout", errorMessage);
        sendEnrollmentStatusSemanticFrame(puid, token, "", false, "ClientTimeout", errorMessage);

        MetricaEvent event{
            .name = "bioAddAccountTimeout",
            .message = errorMessage,
            .puid = puid,
            .timeout = timeout,
        };
        reportMetricaError(std::move(event));

        enrollmentPuid_.reset();
        removeAccount(puid);
    }, addGuestTimeout, lifetime_);
}

void BioCapability::addOwnerAccount() {
    YIO_LOG_INFO("Add owner account");

    Y_ENSURE_THREAD(asyncQueue_);

    if (const auto owner = authProvider_->ownerAuthInfo().value(); owner != nullptr && owner->isAuthorized()) {
        enrollmentPuid_ = owner->passportUid;
        reportMetricaEvent(MetricaEvent{.name = "bioAddAccountSuccess", .puid = owner->passportUid});
        sendGuestEnrollmentStartSemanticFrame(owner->passportUid, owner->authToken);
        return;
    }

    const std::string errorMessage = "No OWNER user";
    YIO_LOG_ERROR_EVENT("BioCapability.AddAccountFailed", errorMessage);
    reportMetricaError(MetricaEvent{.name = "bioAddAccountFailed", .message = errorMessage});
    sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
    onEnrollmentError();
}

void BioCapability::startSoundEnrollment()
{
    if (!bioConfig_.soundSetup) {
        const std::string errorMessage{"Sound enrollment isn't supported"};
        YIO_LOG_ERROR_EVENT("BioCapability.UnsupportedDirective", errorMessage);
        sendEnrollmentStatusSemanticFrame(false, "ScenarioEror", errorMessage);
        onEnrollmentError();
        return;
    }

    device_->telemetry()->reportEvent("bioStartSoundEnrollment");

    YIO_LOG_INFO("Start sound enrollment");

    soundSetup_->startParsing(soundSetupTimeout_);

    soundEnrollmentTimeoutCallback_.executeDelayed([this] {
        const std::string errorMessage{"Sound enrollment timeout"};
        YIO_LOG_ERROR_EVENT("BioCapability.SoundEnrollmentTimeout", errorMessage);
        sendEnrollmentStatusSemanticFrame(false, "ClientTimeout", errorMessage);

        soundSetup_->stopParsing();

        MetricaEvent event{
            .name = "bioSoundEnrollmentTimeout",
            .message = errorMessage,
            .timeout = soundSetupTimeout_,
        };
        reportMetricaError(std::move(event));

        onEnrollmentError();
    }, soundSetupTimeout_, lifetime_);
}

void BioCapability::sendGuestEnrollmentStartSemanticFrame(const std::string& puid, const std::string& token) {
    YIO_LOG_INFO("Send guest enrollment start semantic frame");

    reportMetricaEvent(MetricaEvent{.name = "bioSendEnrollmentSemanticFrame", .puid = puid});

    auto request = VinsRequest::createEventRequest(
        VinsRequest::buildGuestEnrollmentStartSemanticFrame(puid),
        VinsRequest::createSoftwareDirectiveEventSource());

    NAlice::TGuestOptions guestOptions;
    guestOptions.set_oauthtoken(TString(token));
    guestOptions.set_yandexuid(TString(puid));

    request->setGuestOptions(std::move(guestOptions));

    if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
        aliceCapability->startRequest(std::move(request), nullptr);
    }
}

void BioCapability::sendEnrollmentStatusSemanticFrame(bool success, const std::string& failureReason, const std::string& details)
{
    sendEnrollmentStatusSemanticFrame("", "", "", success, failureReason, details);
}

void BioCapability::sendEnrollmentStatusSemanticFrame(const std::string& puid, const std::string& token, const std::string& persId,
                                                      bool success, const std::string& failureReason, const std::string& details) {
    YIO_LOG_INFO("Send enrollment status semantic frame");

    auto request = VinsRequest::createEventRequest(
        VinsRequest::buildEnrollmentStatusSemanticFrame(puid, success, failureReason, details),
        VinsRequest::createSoftwareDirectiveEventSource());

    request->setIsParallel(true);
    request->setIsSilent(true);
    request->setIgnoreAnswer(true);

    NAlice::TGuestOptions guestOptions;

    if (!puid.empty()) {
        guestOptions.set_yandexuid(TString(puid));
    }
    if (!token.empty()) {
        guestOptions.set_oauthtoken(TString(token));
    }
    if (!persId.empty()) {
        guestOptions.set_persid(TString(persId));
    }

    request->setGuestOptions(std::move(guestOptions));

    if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
        aliceCapability->startRequest(std::move(request), nullptr);
    }
}

void BioCapability::sendGuestEnrollmentFinishSemanticFrame(const std::string& puid, const std::string& token, const std::string& persId) {
    YIO_LOG_INFO("Send guest enrollment finish semantic frame");

    auto request = VinsRequest::createEventRequest(
        VinsRequest::buildGuestEnrollmentFinishSemanticFrame(),
        VinsRequest::createSoftwareDirectiveEventSource());

    request->setIsParallel(true);
    request->setIsSilent(true);
    request->setIgnoreAnswer(true);

    NAlice::TGuestOptions guestOptions;
    guestOptions.set_oauthtoken(TString(token));
    guestOptions.set_persid(TString(persId));
    guestOptions.set_yandexuid(TString(puid));

    request->setGuestOptions(std::move(guestOptions));

    if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
        aliceCapability->startRequest(std::move(request), nullptr);
    }
}

void BioCapability::removeAccount(const Json::Value& payload)
{
    Y_ENSURE_THREAD(asyncQueue_);

    device_->telemetry()->reportEvent("bioRemoveAccount");

    std::string puid;

    {
        const auto payloadPuid = tryGetUInt64(payload, "puid", 0);
        if (payloadPuid == 0) {
            const std::string errorMessage{"'puid' is empty. " + jsonToString(payload)};
            MetricaEvent event{
                .name = "bioRemoveAccountFailed",
                .message = errorMessage,
            };
            YIO_LOG_ERROR_EVENT("BioCapability.RemoveAccountFailed", errorMessage);
            sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
            reportMetricaError(std::move(event));
            return;
        }

        puid = std::to_string(payloadPuid);
    }

    const auto foundUserIt = users_.find(puid);
    const PersInfo persInfo = (foundUserIt == users_.end() ? PersInfo{} : foundUserIt->second);
    if (removeAccount(puid)) {
        sendEnrollmentStatusSemanticFrame(puid, persInfo.authToken, persInfo.enrollId,
                                          true, "RequestedByUser", "Account removed");
        reportMetricaEvent(MetricaEvent{.name = "bioRemoveAccountSuccess", .puid = puid});
    }
}

bool BioCapability::removeAccount(const std::string& puid)
{
    Y_ENSURE_THREAD(asyncQueue_);

    const auto userIt = users_.find(puid);
    if (userIt == users_.end()) {
        const std::string errorMessage{"Can't find user"};
        MetricaEvent event{
            .name = "bioRemoveAccountFailed",
            .message = errorMessage,
            .puid = puid,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.RemoveAccountFailed", errorMessage);
        sendEnrollmentStatusSemanticFrame(puid, "", "", false, "ScenarioError", errorMessage);
        reportMetricaError(std::move(event));
        return false;
    }

    const auto accountInfo = userIt->second;

    enrollments_.erase(userIt->second.enrollId);
    userIt->second.enrollId.clear();

    updateEnrollments();
    updateFileStorage();
    updateEnrollmentHeaders();

    if (accountInfo.userType == UserType::GUEST) {
        return removeGuestAccount(puid);
    }

    return true;
}

bool BioCapability::removeGuestAccount(const std::string& puid) {
    YIO_LOG_INFO("Remove guest account");

    bool deleteUserFailed = false;
    std::string errorMessage;

    try {
        const std::int64_t id = std::stoull(puid);
        const std::chrono::seconds timeout{10};
        const auto deletionStatus = authProvider_->deleteUser(id, timeout);
        deleteUserFailed = deletionStatus.status != IAuthProvider::DeleteUserResponse::Status::OK;
        errorMessage = "Can't remove user. Status name: " + deletionStatus.statusName();
    } catch (const std::exception& ex) {
        deleteUserFailed = true;
        errorMessage = "Can't remove user. Exception: " + std::string(ex.what());
    }

    if (deleteUserFailed) {
        MetricaEvent event{
            .name = "bioRemoveAccountFailed",
            .message = errorMessage,
            .puid = puid,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.RemoveAccountFailed", errorMessage);
        sendEnrollmentStatusSemanticFrame(puid, "", "", false, "ScenarioError", errorMessage);
        reportMetricaError(std::move(event));
        return false;
    }

    return true;
}

void BioCapability::startEnrollment(const Json::Value& payload)
{
    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_INFO("Start enrollment");
    device_->telemetry()->reportEvent("bioEnrollmentStart");

    if (enrollmentEngine_->IsEnrollmentActive()) {
        const std::string errorMessage{"Can't start enrollment, due to active enrollment"};
        MetricaEvent event{
            .name = "bioEnrollmentFailed",
            .message = errorMessage,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentStartFailed", errorMessage);
        reportMetricaError(std::move(event));
        sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
        onEnrollmentError();
        return;
    }

    // Owner enrollment is usually initiated by voice. In that case we don't have enrollmentPuid_.
    if (!enrollmentPuid_.has_value()) {
        // This case doesn't seem to be real
        const auto owner = authProvider_->ownerAuthInfo().value();
        if (owner == nullptr) {
            const std::string errorMessage{"Can't start owner enrollment. No owner"};
            MetricaEvent event{
                .name = "bioEnrollmentFailed",
                .message = errorMessage,
            };
            YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentStartFailed", errorMessage);
            reportMetricaError(std::move(event));
            sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
            onEnrollmentError();
            return;
        }

        enrollmentPuid_ = owner->passportUid;
    }

    const auto enrollmentTimeout = tryGetMillis(payload, "timeout_ms", DEFAULT_ENROLLMENT_TIMEOUT);
    YIO_LOG_INFO("Start enrollment with timeout " << enrollmentTimeout.count() << " ms");

    enrollmentEngine_->StartEnrollment();

    reportMetricaEvent(MetricaEvent{.name = "bioEnrollmentStartSuccess", .timeout = enrollmentTimeout});

    cancelEnrollmentCallback_.executeDelayed([this, enrollmentTimeout] {
        PersInfo persInfo;

        if (enrollmentPuid_.has_value()) {
            if (const auto userIt = users_.find(*enrollmentPuid_); userIt != users_.end()) {
                persInfo = userIt->second;
            }
        }

        sendEnrollmentStatusSemanticFrame(enrollmentPuid_.value_or(""), persInfo.authToken, persInfo.enrollId, false, "ClientTimeout", "Enrollment timeout");

        MetricaEvent event{
            .name = "bioEnrollmentTimeout",
            .message = "Cancel enrollment by timeout",
            .timeout = enrollmentTimeout};
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentTimeout", event.message);

        reportMetricaError(std::move(event));
        onCancelEnrollment();
    }, enrollmentTimeout, lifetime_);
}

void BioCapability::cancelEnrollment(const Json::Value& payload)
{
    Y_UNUSED(payload);

    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_INFO("Cancel enrollment");
    device_->telemetry()->reportEvent("bioEnrollmentCancel");

    sendEnrollmentStatusSemanticFrame(enrollmentPuid_.value_or(""), "", "",
                                      false, "ScenarioError", "Cancel enrollment");

    if (!enrollmentEngine_->IsEnrollmentActive()) {
        const std::string errorMessage{"Can't cancel enrollment, no active enrollment"};
        MetricaEvent event{
            .name = "bioEnrollmentFailed",
            .message = errorMessage,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentCancelFailed", errorMessage);
        reportMetricaError(std::move(event));
        return;
    }

    onCancelEnrollment();

    device_->telemetry()->reportEvent("bioEnrollmentCancelSuccess");
}

void BioCapability::onCancelEnrollment()
{
    Y_ENSURE_THREAD(asyncQueue_);

    enrollmentPuid_.reset();
    enrollmentEngine_->CancelEnrollment();

    cancelEnrollmentCallback_.reset();
}

void BioCapability::finishEnrollment(const Json::Value& payload)
{
    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_INFO("Finish enrollment");
    device_->telemetry()->reportEvent("bioEnrollmentFinish");

    if (!enrollmentEngine_->IsEnrollmentActive()) {
        const std::string errorMessage{"No active enrollment"};
        MetricaEvent event{
            .name = "bioEnrollmentFailed",
            .message = errorMessage,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentFinishFailed", errorMessage);
        reportMetricaError(std::move(event));
        sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
        return;
    }

    if (!enrollmentPuid_.has_value()) {
        const std::string errorMessage{"No user for enrollment"};
        MetricaEvent event{
            .name = "bioEnrollmentFailed",
            .message = errorMessage,
        };
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentFinishFailed", errorMessage);
        reportMetricaError(std::move(event));
        sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
        onCancelEnrollment();
        onEnrollmentError();
        return;
    }

    TString enrollId = tryGetString(payload, "pers_id");
    if (enrollId.empty()) {
        const std::string errorMessage{"'pers_id' is empty. " + jsonToString(payload)};
        YIO_LOG_ERROR_EVENT("BioCapability.EnrollmentFinishFailed", errorMessage);

        PersInfo persInfo;

        if (enrollmentPuid_.has_value()) {
            if (const auto userIt = users_.find(*enrollmentPuid_); userIt != users_.end()) {
                persInfo = userIt->second;
            }
        }

        sendEnrollmentStatusSemanticFrame(enrollmentPuid_.value_or(""), persInfo.authToken, persInfo.enrollId, false, "ClientTimeout", errorMessage);

        MetricaEvent event{
            .name = "bioEnrollmentFailed",
            .message = errorMessage,
        };

        reportMetricaError(std::move(event));
        onCancelEnrollment();
        onEnrollmentError();
        return;
    }

    cancelEnrollmentCallback_.reset();
    addGuestTimeoutCallback_.reset();

    auto& user = users_[*enrollmentPuid_];
    user.enrollId = std::move(enrollId);

    enrollments_[user.enrollId] = enrollmentEngine_->CommitEnrollment();

    updateEnrollments();
    updateFileStorage();
    updateEnrollmentHeaders();

    sendEnrollmentStatusSemanticFrame(*enrollmentPuid_, user.authToken, user.enrollId, true, "", "");

    const bool isSendGuestEnrollmentFinishSemanticFrame = tryGetBool(payload, "send_guest_enrollment_finish_frame", false);
    if (isSendGuestEnrollmentFinishSemanticFrame) {
        sendGuestEnrollmentFinishSemanticFrame(*enrollmentPuid_, user.authToken, user.enrollId);
    }

    enrollmentPuid_.reset();

    device_->telemetry()->reportEvent("bioEnrollmentFinishSuccess");
}

void BioCapability::onAliceStateChanged(proto::AliceState state)
{
    if (state.state() == proto::AliceState::LISTENING) {
        startRequest();
    } else {
        finishRequest();
    }
}

void BioCapability::onAliceTtsCompleted()
{
    // Nothing to do
}

void BioCapability::onBioSoundParsingStart(std::chrono::seconds timeout) {
    Y_UNUSED(timeout);
}

void BioCapability::onBioSoundParsingSuccess(const std::string& code) {
    asyncQueue_->add([this, code] {
        addGuestAccount(code, false);
    }, lifetime_);
}

void BioCapability::onBioSoundParsingStop() {
    // Nothing to do
}

void BioCapability::onBioSoundTransferStart() {
    // Nothing to do
}

void BioCapability::onBioSoundSetupError() {
    asyncQueue_->add([this] {
        if (soundSetup_) {
            soundSetup_->stopParsing();
        }

        soundEnrollmentTimeoutCallback_.reset();

        const std::string errorMessage{"Sound enrollment failed"};
        MetricaEvent event{
            .name = "bioSoundEnrollmentFailed",
            .message = errorMessage,
        };

        YIO_LOG_ERROR_EVENT("BioCapability.SoundEnrollmentFailed", errorMessage);
        reportMetricaError(std::move(event));
        sendEnrollmentStatusSemanticFrame(false, "ScenarioError", errorMessage);
        onEnrollmentError();
    }, lifetime_);
}

void BioCapability::startRequest()
{
    asyncQueue_->add([this] {
        if (enrollmentEngine_->IsRequestActive()) {
            return;
        }

        YIO_LOG_INFO("Start request");

        resetMatchedUser();

        forceSendMatchReply_ = true;
        finishRequest_ = false;

        audioListener_->startRequest();
        enrollmentEngine_->StartRequest();
    }, lifetime_);
}

void BioCapability::finishRequest()
{
    asyncQueue_->add([this] {
        if (finishRequest_ || !enrollmentEngine_->IsRequestActive()) {
            return;
        }

        YIO_LOG_INFO("Finish request");

        finishRequest_ = true;
    }, lifetime_);
}

void BioCapability::onSoundBuffer(const ChannelData& soundBuffer)
{
    asyncQueue_->add([this, soundBuffer] {
        if (!enrollmentEngine_->IsRequestActive()) {
            return;
        }

        const TConstArrayRef<unsigned char> chunk{
            reinterpret_cast<const unsigned char*>(soundBuffer.data.data()),
            soundBuffer.data.size() * sizeof(ChannelData::SampleInt)};
        enrollmentEngine_->AddChunk(chunk, finishRequest_);

        sendMatch();

        if (finishRequest_) {
            onLastBufferAction();
            resetMatchedUser();
        }
    }, lifetime_);
}

void BioCapability::onLastBufferAction()
{
    Y_ENSURE_THREAD(asyncQueue_);

    audioListener_->finishRequest();

    auto diagnosticDataJson = enrollmentEngine_->FinishRequest();
    reportBioLibraryDiagnosticData(std::move(diagnosticDataJson));
}

void BioCapability::matchVoicePrint(const Json::Value& payload)
{
    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_DEBUG("Match voice print");
    device_->telemetry()->reportEvent("bioMatchEnroll");

    if (!enrollmentEngine_->IsRequestActive()) {
        YIO_LOG_DEBUG("No active request. Skip match voice print");
        return;
    }

    const auto biometry_result = tryGetString(payload, "biometry_result");
    if (biometry_result.empty()) {
        MetricaEvent event{
            .name = "bioMatchEnrollFailed",
            .message = "'biometry_result' is empty. " + jsonToString(payload),
        };
        YIO_LOG_ERROR_EVENT("BioCapability.MatchVoicePrintFailed", event.message);
        reportMetricaError(std::move(event));
        return;
    }

    enrollmentEngine_->SetRequestExternalVoiceprint(TBlob::FromString(base64Decode(biometry_result)));

    sendMatch();

    forceSendMatchReply_ = false;

    device_->telemetry()->reportEvent("bioMatchEnrollSuccess");
}

bool BioCapability::updateMatchedEnrollmentID()
{
    Y_ENSURE_THREAD(asyncQueue_);

    // It doesn't make sense to get matched enrollment id during active enrollment
    if (enrollmentEngine_->IsEnrollmentActive()) {
        return false;
    }

    // We can't get enrollment id when we don't have active request
    if (!enrollmentEngine_->IsRequestActive()) {
        return false;
    }

    auto matchedId = enrollmentEngine_->GetMatchedEnrollmentID();
    const bool idChanged = matchedEnrollmentId_.has_value() != matchedId.Defined() || (matchedId.Defined() && *matchedId != *matchedEnrollmentId_);

    if (idChanged) {
        matchedEnrollmentId_ = (matchedId.Empty() ? std::nullopt : std::optional(*std::move(matchedId)));
    }

    return idChanged;
}

void BioCapability::sendMatch() {
    Y_ENSURE_THREAD(asyncQueue_);

    const bool sendMatch = updateMatchedEnrollmentID() || (forceSendMatchReply_ && !parentMatchMessageId_.empty());

    if (!sendMatch) {
        return;
    }

    YIO_LOG_INFO("Send match");

    const auto personId = matchedEnrollmentId_.has_value() ? *matchedEnrollmentId_ : "null";
    reportMetricaEvent(MetricaEvent{.name = "bioMatchEnrollChanged", .personId = personId});

    NAlice::TGuestOptions guestOptions;
    guestOptions.set_status(NAlice::TGuestOptions::NoMatch);
    if (matchedEnrollmentId_.has_value()) {
        const auto userIt = std::find_if(users_.cbegin(), users_.cend(), [this](const auto& user) {
            return user.second.enrollId == *matchedEnrollmentId_;
        });
        if (userIt != users_.end()) {
            YIO_LOG_INFO("Guest options: persid=" << userIt->second.enrollId << ", yandexuid=" << userIt->first);
            guestOptions.set_guestorigin(NAlice::TGuestOptions::VoiceBiometry);
            guestOptions.set_oauthtoken(TString(userIt->second.authToken));
            guestOptions.set_yandexuid(TString(userIt->first));
            guestOptions.set_persid(userIt->second.enrollId);
            guestOptions.set_status(NAlice::TGuestOptions::Match);
        } else {
            YIO_LOG_INFO("Guest options: empty");
        }
    }

    auto request = std::make_shared<VinsRequest>(Json::Value(), VinsRequest::createVoiceprintMatchEventSource());
    request->setIsParallel(true);
    request->setIsSilent(true);
    request->setIgnoreAnswer(true);
    request->setParentMessageId(parentMatchMessageId_);
    request->setGuestOptions(std::move(guestOptions));

    if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
        aliceCapability->startRequest(std::move(request), nullptr);
    }
}

void BioCapability::resetMatchedUser()
{
    Y_ENSURE_THREAD(asyncQueue_);

    matchedEnrollmentId_.reset();
    parentMatchMessageId_.clear();
}

void BioCapability::updateVoicePrints(const Json::Value& payload)
{
    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_INFO("Update voice prints");
    device_->telemetry()->reportEvent("bioUpdateEnrolls");

    const auto jsonVoiceprints = tryGetArray(payload, "voiceprints");
    if (!jsonVoiceprints.isArray()) {
        MetricaEvent event{
            .name = "bioUpdateEnrollsFailed",
            .message = "'voiceprints' section is not an array. " + jsonToString(payload),
        };
        YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
        reportMetricaError(std::move(event));
        return;
    }

    bool usersChanged = false;

    for (const auto& jsonVoiceprint : jsonVoiceprints) {
        const auto enrollment = tryGetString(jsonVoiceprint, "enrollment");
        if (enrollment.empty()) {
            MetricaEvent event{
                .name = "bioUpdateEnrollsFailed",
                .message = "'enrollment' is empty. " + jsonToString(jsonVoiceprint),
            };
            YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
            reportMetricaError(std::move(event));
            continue;
        }

        const auto& jsonVoiceprintHeader = tryGetJson(jsonVoiceprint, "voiceprint_header");
        if (!jsonVoiceprintHeader.isObject()) {
            MetricaEvent event{
                .name = "bioUpdateEnrollsFailed",
                .message = "'voiceprint_header' section is not an object. " + jsonToString(jsonVoiceprint),
            };
            YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
            reportMetricaError(std::move(event));
            continue;
        }

        const TString enrollId = tryGetString(jsonVoiceprintHeader, "person_id");

        if (enrollId.empty()) {
            MetricaEvent event{
                .name = "bioUpdateEnrollsFailed",
                .message = "'person_id' field is empty. " + jsonToString(jsonVoiceprintHeader),
            };
            YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
            reportMetricaError(std::move(event));
            continue;
        }

        const auto userType = tryGetString(jsonVoiceprintHeader, "user_type");

        // A special user type to push server voiceprints to devices. Exterminate it when time will come.
        const std::string_view INITIAL_SYNC_SERVER_VOICEPRINTS_USER = "__SYSTEM_OWNER_DO_NOT_USE_AFTER_2021";
        if (userType != INITIAL_SYNC_SERVER_VOICEPRINTS_USER) {
            const auto userIt = std::find_if(users_.cbegin(), users_.cend(), [&enrollId](const auto& persInfo) {
                return persInfo.second.enrollId == enrollId;
            });

            if (userIt == users_.end()) {
                MetricaEvent event{
                    .name = "bioUpdateEnrollsFailed",
                    .message = "Unknown 'person_id'. " + jsonToString(jsonVoiceprintHeader),
                };
                YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
                reportMetricaError(std::move(event));
                continue;
            }

            if (userTypeToString(userIt->second.userType) != userType) {
                MetricaEvent event{
                    .name = "bioUpdateEnrollsFailed",
                    .message = "Invalid 'user_type'. " + jsonToString(jsonVoiceprintHeader),
                };
                YIO_LOG_ERROR_EVENT("BioCapability.UpdateVoicePrintsFailed", event.message);
                reportMetricaError(std::move(event));
                continue;
            }

            enrollments_[enrollId] = TBlob::FromString(base64Decode(enrollment));
        } else {
            const auto owner = authProvider_->ownerAuthInfo().value();
            users_[owner->passportUid].enrollId = enrollId;
            users_[owner->passportUid].authToken = owner->authToken;
            users_[owner->passportUid].userType = UserType::OWNER;
            enrollments_[enrollId] = TBlob::FromString(base64Decode(enrollment));
        }

        usersChanged = true;
    }

    if (usersChanged) {
        updateEnrollments();
        updateFileStorage();
        updateEnrollmentHeaders();
        device_->telemetry()->reportEvent("bioUpdateEnrollsSuccess");
    }
}

void BioCapability::updateEnrollmentHeaders() const {
    Y_ENSURE_THREAD(asyncQueue_);

    YIO_LOG_INFO("Update enrollment headers");

    auto versionsInfo = enrollmentEngine_->GetEnrollmentsVersionInfo();

    NAlice::TEnrollmentHeaders enrollmentHeaders;
    for (const auto& [puid, user] : users_) {
        TString enrollVersion;

        const auto enrollmentVersionIt = versionsInfo.find(user.enrollId);
        if (enrollmentVersionIt != versionsInfo.end()) {
            enrollVersion = enrollmentVersionIt->second;
        }

        auto header = enrollmentHeaders.add_headers();
        header->set_version(std::move(enrollVersion));
        header->set_personid(user.enrollId);
        header->set_usertype(userTypeToProto(user.userType));
    }

    device_->telemetry()->reportEvent("bioEnrollmentHeadersChanged", convertMessageToJsonString(enrollmentHeaders, true));

    if (const auto deviceStateCapability = sdk_->getDeviceStateCapability(); deviceStateCapability != nullptr) {
        deviceStateCapability->setEnrollmentHeaders(enrollmentHeaders);
    }
}

void BioCapability::onEnrollmentError() {
    if (const auto filePlayerCapability = sdk_->getFilePlayerCapability(); filePlayerCapability != nullptr) {
        filePlayerCapability->playSoundFile("guest_enrollment_failed.mp3", proto::DIALOG_CHANNEL);
    }
}

Json::Value BioCapability::MetricaEvent::toJson() const {
    Json::Value json;
    if (!message.empty()) {
        json["message"] = TString(message);
    }
    if (timeout.has_value()) {
        json["timeout_ms"] = static_cast<Json::UInt64>(timeout->count());
    }
    if (puid.has_value()) {
        json["puid"] = TString(*puid);
    }
    if (personId.has_value()) {
        json["person_id"] = *personId;
    }
    return json;
}

void BioCapability::reportMetricaEvent(MetricaEvent event)
{
    device_->telemetry()->reportEvent(event.name, jsonToString(event.toJson()));
}

void BioCapability::reportMetricaError(MetricaEvent event)
{
    device_->telemetry()->reportError(event.name, jsonToString(event.toJson()));
}

void BioCapability::reportBioLibraryDiagnosticData(NJson::TJsonValue json)
{
    // Workaround to find an invalid value
    const NJson::TJsonWriterConfig jsonWriterConfig{
        .WriteNanAsString = true,
    };

    TStringStream ss;
    auto jsonWriter = NJson::TJsonWriter(&ss, jsonWriterConfig, false);
    jsonWriter.Write(&json);
    jsonWriter.Flush();

    const auto diagnosticData = std::string(ss.Str());
    YIO_LOG_INFO("Diagnostic data: " << diagnosticData);
    device_->telemetry()->reportEvent("bioLibraryDiagnosticData", diagnosticData);
}
