#pragma once

#include "sound_setup/bio_sound_setup.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>

#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/device_cryptography/device_cryptography.h>
#include <yandex_io/libs/threading/unique_callback.h>
#include <yandex_io/libs/threading/i_callback_queue.h>

#include <yandex_io/sdk/sdk_interface.h>

#include <yandex_io/sdk/audio_source/i_audio_source_client.h>
#include <yandex_io/sdk/interfaces/i_capability.h>
#include <yandex_io/sdk/interfaces/i_directive_handler.h>

#include <yandex_io/capabilities/alice/interfaces/i_alice_capability.h>
#include <yandex_io/capabilities/alice/interfaces/i_alice_capability_listener.h>
#include <yandex_io/capabilities/device_state/interfaces/i_device_state_capability.h>
#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>

#include <alice/protos/endpoint/capabilities/bio/capability.pb.h>

#include <voicetech/bio/ondevice/lib/engine/enrollment_engine.h>

#include <json/json-forwards.h>

#include <atomic>
#include <functional>
#include <memory>
#include <optional>
#include <string>

namespace YandexIO {

    class BioSoundSetup;

    struct BioConfig {
        std::string modelPath;
        std::string storagePath;
        bool soundSetup = false;
    };

    class BioCapability
        : public ICapability,
          public IDirectiveHandler,
          public IAliceCapabilityListener,
          public BioSoundSetup::IListener,
          public std::enable_shared_from_this<BioCapability> {
    public:
        static std::unique_ptr<BioCapability> create(
            std::shared_ptr<IDevice> device,
            std::shared_ptr<SDKInterface> sdk,
            std::shared_ptr<quasar::IAuthProvider> authProvider,
            std::shared_ptr<IAudioSourceClient> audioSourceClient,
            BioConfig bioConfig);

        // Constructor for tests
        BioCapability(std::shared_ptr<IDevice> device,
                      std::shared_ptr<SDKInterface> sdk,
                      std::shared_ptr<quasar::IAuthProvider> authProvider,
                      std::shared_ptr<IAudioSourceClient> audioSource,
                      std::shared_ptr<NBio::NDevice::IEnrollmentEngine> enrollmentEngine,
                      BioConfig config,
                      std::shared_ptr<quasar::ICallbackQueue> asyncQueue);
        ~BioCapability() override;

        void setBioSoundSetup(std::unique_ptr<BioSoundSetup> soundSetup);

        // ICapability implementation
        NAlice::TCapabilityHolder getState() const override;
        YandexIO::IDirectiveHandlerPtr getDirectiveHandler() override;

        void addListener(std::weak_ptr<ICapability::IListener> wlistener) override;
        void removeListener(std::weak_ptr<ICapability::IListener> wlistener) override;

        // IDirectiveHandler
        const std::string& getEndpointId() const override;
        const std::string& getHandlerName() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void handleDirective(const std::shared_ptr<Directive>& directive) override;
        void cancelDirective(const std::shared_ptr<Directive>& directive) override;
        void prefetchDirective(const std::shared_ptr<Directive>& directive) override;

        void onSoundBuffer(const ChannelData& soundBuffer);

        // IAliceCapabilityListener
        void onAliceStateChanged(quasar::proto::AliceState state) override;
        void onAliceTtsCompleted() override;

        // BioSoundSetup::IListener
        void onBioSoundParsingStart(std::chrono::seconds timeout) override;
        void onBioSoundParsingSuccess(const std::string& code) override;
        void onBioSoundParsingStop() override;
        void onBioSoundTransferStart() override;
        void onBioSoundSetupError() override;

    private:
        void onDeleteGuestAccount();

        void onAccountsChanged(const quasar::IAuthProvider::UsersAuthInfo& usersAuthInfo);

        void startRequest();
        void finishRequest();

        void startEnrollment(const Json::Value& payload);
        void finishEnrollment(const Json::Value& payload);
        void cancelEnrollment(const Json::Value& payload);

        void onCancelEnrollment();

        void startSoundEnrollment();

        void init();

        void loadFileStorage();

        void updateFileStorage() const;
        void updateEnrollments();
        void updateEnrollmentHeaders() const;

        void addAccount(const Json::Value& payload);

        void addOwnerAccount();

        void addGuestAccount(const Json::Value& payload);
        void addGuestAccount(const std::string& authCode, bool withXToken);

        void sendGuestEnrollmentStartSemanticFrame(const std::string& puid, const std::string& token);
        void sendEnrollmentStatusSemanticFrame(bool success, const std::string& failureReason, const std::string& details);
        void sendEnrollmentStatusSemanticFrame(const std::string& puid, const std::string& token, const std::string& persId,
                                               bool success, const std::string& failureReason, const std::string& details);
        void sendGuestEnrollmentFinishSemanticFrame(const std::string& puid, const std::string& token, const std::string& persId);

        void removeAccount(const Json::Value& payload);
        bool removeAccount(const std::string& puid);
        bool removeGuestAccount(const std::string& puid);

        void updateVoicePrints(const Json::Value& payload);
        void matchVoicePrint(const Json::Value& payload);

        void sendMatch();
        bool updateMatchedEnrollmentID();

        void onLastBufferAction();

        void resetMatchedUser();

        struct MetricaEvent {
            std::string name;
            std::string message;
            std::optional<std::string> puid = std::nullopt;
            std::optional<TString> personId = std::nullopt;
            std::optional<std::chrono::milliseconds> timeout = std::nullopt;

            Json::Value toJson() const;
        };
        void reportMetricaEvent(MetricaEvent event);
        void reportMetricaError(MetricaEvent event);
        void reportBioLibraryDiagnosticData(NJson::TJsonValue json);

        void onEnrollmentError();

        void initSupportedDirectiveNames();

    private:
        struct RequestListener: public IAudioSourceClient::Listener,
                                public std::enable_shared_from_this<IAudioSourceClient::Listener> {
            explicit RequestListener(BioCapability& bioCapability);

            void startRequest();
            void finishRequest();

            void onAudioData(const ChannelsData& channels) override;

        private:
            BioCapability& bioCapability_;
            std::atomic<bool> isRequestActive_;
        };

        struct PersInfo {
            TString enrollId;
            std::string authToken;
            quasar::UserType userType = quasar::UserType::GUEST;

            static PersInfo fromJson(const Json::Value& /*json*/);
            Json::Value toJson() const;
        };

    private:
        const std::shared_ptr<IDevice> device_;
        const std::shared_ptr<YandexIO::SDKInterface> sdk_;

        std::unique_ptr<DeviceCryptography> deviceCryptography_;
        std::unique_ptr<BioSoundSetup> soundSetup_;
        const std::chrono::seconds soundSetupTimeout_{60};

        std::shared_ptr<NBio::NDevice::IEnrollmentEngine> enrollmentEngine_;
        std::shared_ptr<quasar::IAuthProvider> authProvider_;
        std::shared_ptr<IAudioSourceClient> audioSource_;
        std::shared_ptr<RequestListener> audioListener_;

        BioConfig bioConfig_;
        std::set<std::string> supportedDirectiveNames_;

        NBio::NDevice::IEnrollmentEngine::TEnrollments enrollments_;
        std::unordered_map<std::string, PersInfo> users_;

        std::optional<std::string> enrollmentPuid_;
        std::optional<TString> matchedEnrollmentId_;
        std::string parentMatchMessageId_;
        bool forceSendMatchReply_ = false;

        NAlice::TCapabilityHolder capabilityState_;
        bool finishRequest_ = false;

        quasar::Lifetime lifetime_;
        std::shared_ptr<quasar::ICallbackQueue> asyncQueue_;
        quasar::UniqueCallback cancelEnrollmentCallback_;
        quasar::UniqueCallback addGuestTimeoutCallback_;
        quasar::UniqueCallback soundEnrollmentTimeoutCallback_;
    };

} // namespace YandexIO
