#include "bio_capability_switcher.h"

#include "bio_capability.h"
#include "sound_setup/bio_sound_setup.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("BioCapability");

using namespace quasar;
using namespace YandexIO;

namespace {

    const std::string CONFIG_NAME = "bioCapabilityEnabled";

    BioConfig readBioConfig(const std::shared_ptr<IDevice>& device) {
        const auto bioConfigJson = device->configuration()->getServiceConfig("personalization");
        return BioConfig{
            .modelPath = tryGetString(bioConfigJson, "modelPath"),
            .storagePath = tryGetString(bioConfigJson, "storagePath"),
            .soundSetup = tryGetBool(bioConfigJson, "soundSetup", false),
        };
    }
} // namespace

BioCapabilitySwitcher::BioCapabilitySwitcher(
    std::shared_ptr<SDKInterface> sdk,
    std::shared_ptr<IDevice> device,
    std::shared_ptr<BioSoundSetup::IListener> soundSetupListener,
    std::shared_ptr<IEndpoint> localEndpoint,
    std::shared_ptr<quasar::IAuthProvider> authProvider,
    std::shared_ptr<IAudioSourceClient> audioSourceClient)
    : sdk_(std::move(sdk))
    , device_(std::move(device))
    , localEndpoint_(std::move(localEndpoint))
    , authProvider_(std::move(authProvider))
    , audioSourceClient_(std::move(audioSourceClient))
    , soundSetupListener_(std::move(soundSetupListener))
    , bioConfig_(readBioConfig(device_))
{
    sdk_->subscribeToSystemConfig(CONFIG_NAME);
}

void BioCapabilitySwitcher::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue)
{
    if (configName != CONFIG_NAME) {
        return;
    }

    const auto json = tryParseJsonOrEmpty(jsonConfigValue);
    const bool configValue = json.isBool() ? json.asBool() : false;

    if (enabled_ != configValue) {
        enabled_ = configValue;
        if (enabled_) {
            YIO_LOG_INFO("Enable BioCapability");

            bioCapability_ = BioCapability::create(device_, sdk_, authProvider_, audioSourceClient_, bioConfig_);
            if (bioConfig_.soundSetup) {
                auto soundSetup = std::make_unique<BioSoundSetup>(device_, sdk_, audioSourceClient_);
                soundSetup->addListener(soundSetupListener_);
                soundSetup->addListener(bioCapability_);

                bioCapability_->setBioSoundSetup(std::move(soundSetup));
            }

            if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
                aliceCapability->addListener(bioCapability_);
            }
            localEndpoint_->addCapability(bioCapability_);
        } else {
            YIO_LOG_INFO("Disable BioCapability");
            if (const auto aliceCapability = sdk_->getAliceCapability(); aliceCapability != nullptr) {
                aliceCapability->removeListener(bioCapability_);
            }
            localEndpoint_->removeCapability(bioCapability_);
            bioCapability_.reset();
        }
    }
}
