#include "bio_sound_setup.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/setup_parser/credentials.h>
#include <yandex_io/libs/setup_parser/setup_parser.h>
#include <yandex_io/libs/setup_parser/sound_utils.h>

#include <yandex_io/modules/audio_input/callback_listener/audio_source_callback_listener.h>
#include <yandex_io/modules/personalization/bio_capability.h>

YIO_DEFINE_LOG_MODULE("BioSoundSetup");

using namespace quasar;
using namespace YandexIO;

BioSoundSetup::BioSoundSetup(
    std::shared_ptr<IDevice> device,
    std::shared_ptr<SDKInterface> sdk,
    std::shared_ptr<IAudioSourceClient> audioClient)
    : voiceAssistantBlocker_{sdk, "BioSoundSetup"}
    , filePlayerCapability_{sdk->getFilePlayerCapability()}
    , audioClient_{std::move(audioClient)}
    , audioListener_{std::make_shared<AudioSourceCallbackListener>(getOnAudioDataCallback())}
    , soundDataReceiver_{std::move(device), SoundUtils::DEFAULT_SAMPLE_RATE}
{
    soundDataReceiver_.onDataReceived = std::bind(&BioSoundSetup::onDataReceived, this, std::placeholders::_1, std::placeholders::_2);
    soundDataReceiver_.onUnsupportedProtocol = std::bind(&BioSoundSetup::onUnsupportedProtocol, this, std::placeholders::_1);
    soundDataReceiver_.onTransferStart = std::bind(&BioSoundSetup::onTransferStart, this);
    soundDataReceiver_.onTransferError = std::bind(&BioSoundSetup::onTransferError, this);

    audioClient_->addListener(audioListener_);
}

BioSoundSetup::OnAudioData BioSoundSetup::getOnAudioDataCallback() {
    return [this](const ChannelsData& channels) {
        if (channels.empty()) {
            return;
        }

        if (!isActive_.load()) {
            return;
        }

        const auto it = std::find_if(channels.cbegin(), channels.cend(), [](const ChannelData& channel) {
            return channel.isForRecognition;
        });

        const auto& channelData = it != channels.end() ? it->data : channels.front().data;

        soundDataReceiver_.write(channelData);
    };
}

void BioSoundSetup::addListener(std::weak_ptr<IListener> listener)
{
    listeners_.insert(listener);
}

void BioSoundSetup::startParsing(std::chrono::seconds timeout)
{
    isActive_.store(true);
    soundDataReceiver_.start();

    for (auto ilistener : listeners_) {
        if (auto listener = ilistener.lock()) {
            listener->onBioSoundParsingStart(timeout);
        }
    }

    voiceAssistantBlocker_.block();
}

void BioSoundSetup::stopParsing()
{
    isActive_.store(false);
    soundDataReceiver_.stop();

    for (auto ilistener : listeners_) {
        if (auto listener = ilistener.lock()) {
            listener->onBioSoundParsingStop();
        }
    }

    voiceAssistantBlocker_.unblock();
}

void BioSoundSetup::onDataReceived(const std::vector<unsigned char>& payload, int protocolVersion)
{
    // Stop feeding sound to soundDataReceiver_
    isActive_.store(false);

    if (protocolVersion < 1) {
        onUnsupportedProtocol(protocolVersion);
        return;
    }

    try {
        const auto credentials = SetupParser::parseInitData(payload);
        YIO_LOG_INFO("Parsed token code: " << credentials.tokenCode);

        for (auto ilistener : listeners_) {
            if (auto listener = ilistener.lock()) {
                listener->onBioSoundParsingSuccess(credentials.tokenCode);
            }
        }
    } catch (std::exception& ex) {
        YIO_LOG_ERROR_EVENT("BioSoundSetup.SetupParserException", ex.what());
        for (auto ilistener : listeners_) {
            if (auto listener = ilistener.lock()) {
                listener->onBioSoundSetupError();
            }
        }
    }
}

void BioSoundSetup::onUnsupportedProtocol(int version)
{
    Json::Value json;
    json["version"] = version;
    YIO_LOG_ERROR_EVENT("BioSoundSetup.UnsupportedProtocol", jsonToString(json));

    for (auto ilistener : listeners_) {
        if (auto listener = ilistener.lock()) {
            listener->onBioSoundSetupError();
        }
    }

    if (const auto filePlayerCapability = filePlayerCapability_.lock(); filePlayerCapability != nullptr) {
        filePlayerCapability->playSoundFile("wrong_sound_version.wav", proto::DIALOG_CHANNEL);
    }
}

void BioSoundSetup::onTransferStart()
{
    for (auto ilistener : listeners_) {
        if (auto listener = ilistener.lock()) {
            listener->onBioSoundTransferStart();
        }
    }
}

void BioSoundSetup::onTransferError()
{
    for (auto ilistener : listeners_) {
        if (auto listener = ilistener.lock()) {
            listener->onBioSoundSetupError();
        }
    }

    if (const auto filePlayerCapability = filePlayerCapability_.lock(); filePlayerCapability != nullptr) {
        filePlayerCapability->playSoundFile("guest_enrollment_failed.mp3", proto::DIALOG_CHANNEL);
    }
}
