#include "qc_mode_handler.h"

#include <yandex_io/modules/leds/led_manager/ng/default_animation_conductor.h>

QCModeHandler::QCModeHandler(std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability,
                             std::shared_ptr<quasar::LedManager> ledManager,
                             std::shared_ptr<YandexIO::BluetoothController> bluetoothController,
                             std::shared_ptr<YandexIO::VersionProvider> versionProvider,
                             std::optional<std::string> buttonsTestSound)
    : QCModeHandler(filePlayerCapability, ledManager, bluetoothController, versionProvider, std::make_shared<AnimationComposition>(std::vector<std::shared_ptr<Animation>>{ledManager->getPattern("qc.led")}, false), buttonsTestSound)
{
}

QCModeHandler::QCModeHandler(std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability,
                             std::shared_ptr<quasar::LedManager> ledManager,
                             std::shared_ptr<YandexIO::BluetoothController> bluetoothController,
                             std::shared_ptr<YandexIO::VersionProvider> versionProvider,
                             std::shared_ptr<AnimationComposition> qcAnimationComposition,
                             std::optional<std::string> buttonsTestSound)
    : filePlayerCapability_(std::move(filePlayerCapability))
    , ledManager_(std::move(ledManager))
    , bluetoothController_(std::move(bluetoothController))
    , versionProvider_(std::move(versionProvider))
    , qcAnimationComposition_(std::move(qcAnimationComposition))
    , buttonsTestSound_(std::move(buttonsTestSound))
{
}

void QCModeHandler::startQCMode(QCMode mode)
{
    {
        std::lock_guard<std::mutex> lock(mutex_);
        if (mode == QCMode::LEDS_TEST) {
            auto foregroundConductor = std::make_shared<DefaultAnimationConductor>(qcAnimationComposition_);
            foregroundConductor->setSubstitutionType(SubstitutionType::SUBSTITUTION);
            ledManager_->play(std::move(foregroundConductor));
        } else if (mode == QCMode::BUTTONS_TEST && buttonsTestSound_.has_value()) {
            YandexIO::IFilePlayerCapability::PlayParams params;
            params.playLooped = true;

            filePlayerCapability_->playSoundFile(buttonsTestSound_.value(), std::nullopt, std::move(params));
        } else if (mode == QCMode::SWEEP_TONE) {
            if (quasar::generateSweepTone(SWEEP_TONE_PATH)) {
                /// Temporary solution, until enter_configure.wav is not playing in DIALOG_CHANNEL
                filePlayerCapability_->stopSoundFile("enter_configure.wav");
                filePlayerCapability_->playSoundFile(SWEEP_TONE_PATH, quasar::proto::AudioChannel::DIALOG_CHANNEL);
            }
        } else if (mode == QCMode::BLUETOOTH) {
            bluetoothController_->setVisibility(true, true);
        } else if (mode == QCMode::PROVIDE_VERSION) {
            versionProvider_->provideVersion();
        }
    }

    for (const auto& listener : listeners_) {
        listener->onQCModeStarted(mode);
    }
}

void QCModeHandler::stopQCMode(QCMode mode)
{
    {
        std::lock_guard<std::mutex> lock(mutex_);
        if (mode == QCMode::BUTTONS_TEST) {
            filePlayerCapability_->stopSoundFile(buttonsTestSound_.value());
        }
    }

    for (const auto& listener : listeners_) {
        listener->onQCModeStopped(mode);
    }
}

void QCModeHandler::addListener(std::shared_ptr<Listener> listener)
{
    listeners_.push_back(std::move(listener));
}

void QCModeHandler::onCommand(std::string_view command)
{
    if (command == "qc_leds_test") {
        startQCMode(QCMode::LEDS_TEST);
    } else if (command == "qc_buttons_test") {
        startQCMode(QCMode::BUTTONS_TEST);
    } else if (command == "qc_sweep_tone") {
        startQCMode(QCMode::SWEEP_TONE);
    } else if (command == "qc_bt_discoverable_on") {
        startQCMode(QCMode::BLUETOOTH);
    } else if (command == "qc_send_version_to_backend") {
        startQCMode(QCMode::PROVIDE_VERSION);
    }
}
