#include "smart_volume_configurer.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("SmartVolume");

using namespace quasar;
using namespace YandexIO;

namespace {

    constexpr auto const CONFIG_NAME = "smart_volume";

    std::map<int, int> parseRmsVolumeMap(const Json::Value& rmsToVolume) {
        std::map<int, int> rmsVolumeMap;

        try {
            for (auto it = rmsToVolume.begin(); it != rmsToVolume.end(); ++it) {
                const int rms = std::stoi(it.name());
                const int volume = it->asInt();
                rmsVolumeMap.emplace(rms, volume);
            }
        } catch (const std::exception& ex) {
            YIO_LOG_INFO("Couldn't parse RMS volume map. " << ex.what());
            return {};
        }

        return rmsVolumeMap;
    }

    SmartVolumeManager::Config parseConfig(const Json::Value& config) {
        const auto rmsToVolumeJson = tryGetJson(config, "rmsToVolume");

        return SmartVolumeManager::Config{
            .rmsVolumeMap = parseRmsVolumeMap(rmsToVolumeJson),
            .useVqeChannel = tryGetBool(config, "useVQEChannel", false),
            .useVoiceFilter = tryGetBool(config, "useVoiceFilter", false),
        };
    }

} // namespace

std::shared_ptr<SmartVolumeConfigurer> SmartVolumeConfigurer::create(std::shared_ptr<SDKInterface> sdk, std::shared_ptr<VolumeManager> volumeManager, std::shared_ptr<IAudioSourceClient> audioSource)
{
    auto configurer = std::shared_ptr<SmartVolumeConfigurer>(
        new SmartVolumeConfigurer(sdk, std::move(volumeManager), std::move(audioSource)));
    sdk->addBackendConfigObserver(configurer);
    return configurer;
}

SmartVolumeConfigurer::SmartVolumeConfigurer(std::shared_ptr<SDKInterface> sdk, std::shared_ptr<VolumeManager> volumeManager, std::shared_ptr<IAudioSourceClient> audioSource)
    : sdk_{std::move(sdk)}
    , volumeManager_{std::move(volumeManager)}
    , audioSource_{std::move(audioSource)}
{
    sdk_->subscribeToDeviceConfig(CONFIG_NAME);
    sdk_->subscribeToSystemConfig(CONFIG_NAME);
}

void SmartVolumeConfigurer::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue)
{
    if (configName != CONFIG_NAME) {
        return;
    }

    // TODO: On/off is a per-device setting
    const auto jsonConfig = tryParseJsonOrEmpty(jsonConfigValue);

    const bool enabled = tryGetBool(jsonConfig, "enabled", false);
    const auto newConfig = parseConfig(jsonConfig);

    if (newConfig.rmsVolumeMap.empty()) {
        YIO_LOG_INFO("Ignore invalid config");
        return;
    }

    if (enabled_ != enabled) {
        if (enabled) {
            YIO_LOG_INFO("Enable SmartVolume");
            smartVolumeManager_ = std::make_shared<SmartVolumeManager>(volumeManager_, audioSource_, newConfig);
            if (auto aliceCapability = sdk_->getAliceCapability()) {
                aliceCapability->addListener(smartVolumeManager_);
            }
        } else {
            YIO_LOG_INFO("Disable SmartVolume");
            auto smartVolumeManager = std::move(smartVolumeManager_);
            if (auto aliceCapability = sdk_->getAliceCapability()) {
                aliceCapability->removeListener(smartVolumeManager);
            }
        }

        enabled_ = enabled;
        config_ = newConfig;
    }

    if (enabled_ && config_ != newConfig) {
        config_ = newConfig;
        smartVolumeManager_->updateConfig(config_);
    }
}

void SmartVolumeConfigurer::onDeviceConfig(const std::string& /* configName */, const std::string& /* jsonConfigValue */)
{
}
