#include "smart_volume_manager.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <ios>
#include <iostream>

YIO_DEFINE_LOG_MODULE("SmartVolume");

using namespace quasar;
using namespace YandexIO;

namespace {

    std::ostream& operator<<(std::ostream& os, const SmartVolumeManager::Config& config)
    {
        os << "Use VQE channel: " << std::boolalpha << config.useVqeChannel << ", use voice filter: " << config.useVoiceFilter << ", RMS to volume map:";
        for (const auto& [rms, volume] : config.rmsVolumeMap) {
            os << " (" << rms << "," << volume << ")";
        }
        return os;
    }
} // namespace

SmartVolumeManager::RMSSource::RMSSource(const SmartVolumeManager::Config& config)
    : voiceStats{quasar::VoiceStats::create()}
    , useRmsWeights{config.useVoiceFilter}
    , rmsWeights{
          0.0671, 0.0700, 0.0735, 0.0753, 0.0743, 0.0716, 0.0702, 0.0728, 0.0807, 0.0944, 0.1137,
          0.1390, 0.1711, 0.2101, 0.2542, 0.3000, 0.3436, 0.3825, 0.4157, 0.4441, 0.4685, 0.4888,
          0.5044, 0.5145, 0.5200, 0.5245, 0.5339, 0.5555, 0.5951, 0.6548, 0.7315, 0.8166, 0.8982,
          0.9642, 1.0051, 1.0162, 0.9978, 0.9549, 0.8951, 0.8280, 0.7630, 0.7078, 0.6668, 0.6399,
          0.6237, 0.6132, 0.6040, 0.5937, 0.5821, 0.5704, 0.5601, 0.5528, 0.5498, 0.5519, 0.5588,
          0.5684, 0.5775, 0.5828, 0.5830, 0.5792, 0.5745, 0.5718, 0.5720, 0.5739, 0.5755, 0.5752,
          0.5730, 0.5700, 0.5672, 0.5649, 0.5628, 0.5610, 0.5602, 0.5611, 0.5637, 0.5665, 0.5668,
          0.5625, 0.5524, 0.5375, 0.5197, 0.5011, 0.4832, 0.4666, 0.4511, 0.4363, 0.4216, 0.4068,
          0.3921, 0.3786, 0.3676, 0.3603, 0.3571, 0.3573, 0.3598, 0.3631, 0.3664, 0.3694, 0.3716,
          0.3726, 0.3721, 0.3698, 0.3662, 0.3622, 0.3588, 0.3566, 0.3557, 0.3560, 0.3571, 0.3587,
          0.3605, 0.3621, 0.3631, 0.3633, 0.3628, 0.3617, 0.3600, 0.3573, 0.3538, 0.3498, 0.3465,
          0.3455, 0.3475, 0.3525, 0.3593, 0.3662, 0.3717, 0.3749, 0.3755, 0.3736, 0.3699, 0.3652,
          0.3604, 0.3560, 0.3523, 0.3491, 0.3456, 0.3417, 0.3378, 0.3348, 0.3340, 0.3358, 0.3395,
          0.3438, 0.3474, 0.3490, 0.3483, 0.3454, 0.3405, 0.3339, 0.3259, 0.3173, 0.3088, 0.3013,
          0.2951, 0.2903, 0.2864, 0.2832, 0.2803, 0.2777, 0.2755, 0.2737, 0.2723, 0.2714, 0.2710,
          0.2713, 0.2722, 0.2738, 0.2755, 0.2771, 0.2783, 0.2788, 0.2788, 0.2785, 0.2780, 0.2774,
          0.2767, 0.2758, 0.2746, 0.2731, 0.2710, 0.2685, 0.2656, 0.2625, 0.2594, 0.2564, 0.2538,
          0.2516, 0.2496, 0.2480, 0.2466, 0.2456, 0.2450, 0.2445, 0.2442, 0.2438, 0.2437, 0.2439,
          0.2445, 0.2453, 0.2458, 0.2452, 0.2435, 0.2406, 0.2373, 0.2342, 0.2315, 0.2293, 0.2273,
          0.2254, 0.2233, 0.2211, 0.2190, 0.2172, 0.2159, 0.2154, 0.2157, 0.2162, 0.2166, 0.2163,
          0.2153, 0.2140, 0.2129, 0.2123, 0.2118, 0.2111, 0.2097, 0.2077, 0.2054, 0.2037, 0.2030,
          0.2032, 0.2041, 0.2054, 0.2068, 0.2082, 0.2094, 0.2104, 0.2108, 0.2110, 0.2112, 0.2119,
          0.2131, 0.2146, 0.2156, 0.2156, 0.2145, 0.2127, 0.2111, 0.2099, 0.2093, 0.2087, 0.2078,
          0.2061, 0.2038, 0.2012, 0.1984, 0.1959, 0.1937, 0.1921, 0.1911, 0.1907, 0.1907, 0.1907,
          0.1904, 0.1897, 0.1889, 0.1881, 0.1877, 0.1874, 0.1872, 0.1866, 0.1855, 0.1841, 0.1827,
          0.1815, 0.1807, 0.1801, 0.1797, 0.1796, 0.1797, 0.1800, 0.1805, 0.1809, 0.1810, 0.1810,
          0.1810, 0.1815, 0.1824, 0.1837, 0.1849, 0.1857, 0.1861, 0.1865, 0.1872, 0.1883, 0.1895,
          0.1903, 0.1903, 0.1895, 0.1880, 0.1862, 0.1847, 0.1837, 0.1833, 0.1833, 0.1835, 0.1838,
          0.1841, 0.1841, 0.1838, 0.1834, 0.1827, 0.1819, 0.1810, 0.1801, 0.1794, 0.1789, 0.1785,
          0.1780, 0.1773, 0.1762, 0.1748, 0.1734, 0.1723, 0.1718, 0.1718, 0.1722, 0.1729, 0.1736,
          0.1743, 0.1748, 0.1751, 0.1751, 0.1751, 0.1754, 0.1762, 0.1773, 0.1783, 0.1788, 0.1786,
          0.1779, 0.1772, 0.1767, 0.1765, 0.1765, 0.1762, 0.1756, 0.1749, 0.1745, 0.1746, 0.1753,
          0.1764, 0.1775, 0.1782, 0.1784, 0.1780, 0.1772, 0.1762, 0.1755, 0.1752, 0.1752, 0.1754,
          0.1755, 0.1754, 0.1754, 0.1754, 0.1754, 0.1753, 0.1748, 0.1738, 0.1724, 0.1711, 0.1701,
          0.1694, 0.1690, 0.1685, 0.1677, 0.1666, 0.1654, 0.1643, 0.1633, 0.1624, 0.1616, 0.1610,
          0.1607, 0.1609, 0.1616, 0.1626, 0.1637, 0.1645, 0.1650, 0.1652, 0.1650, 0.1646, 0.1642,
          0.1639, 0.1638, 0.1638, 0.1640, 0.1641, 0.1642, 0.1643, 0.1644, 0.1647, 0.1651, 0.1656,
          0.1662, 0.1668, 0.1674, 0.1679, 0.1684, 0.1690, 0.1698, 0.1708, 0.1718, 0.1728, 0.1734,
          0.1734, 0.1731, 0.1725, 0.1718, 0.1712, 0.1705, 0.1696, 0.1686, 0.1677, 0.1672, 0.1672,
          0.1677, 0.1682, 0.1683, 0.1678, 0.1665, 0.1650, 0.1635, 0.1623, 0.1616, 0.1613, 0.1609,
          0.1605, 0.1599, 0.1592, 0.1587, 0.1582, 0.1577, 0.1570, 0.1560, 0.1548, 0.1537, 0.1527,
          0.1519, 0.1513, 0.1508, 0.1501, 0.1495, 0.1491, 0.1488, 0.1488, 0.1488, 0.1488, 0.1488,
          0.1489, 0.1490, 0.1493, 0.1498, 0.1503, 0.1508, 0.1514, 0.1520, 0.1526, 0.1530, 0.1532,
          0.1530, 0.1524, 0.1515, 0.1504, 0.1494, 0.1486, 0.1480, 0.1474, 0.1468, 0.1460, 0.1450,
          0.1439, 0.1429, 0.1420, 0.1412, 0.1401, 0.1385, 0.1365, 0.1344, 0.1326, 0.1314, 0.1306,
          0.1301, 0.1295, 0.1286, 0.1274, 0.1262, 0.1251, 0.1242, 0.1234, 0.1227, 0.1221, 0.1216,
          0.1210, 0.1204, 0.1197, 0.1189, 0.1182, 0.1177, 0.1174, 0.1172, 0.1170, 0.1168, 0.1165,
          0.1162, 0.1158, 0.1153, 0.1150, 0.1146, 0.1143, 0.1139, 0.1133, 0.1125, 0.1114, 0.1102,
          0.1092, 0.1084, 0.1078, 0.1073, 0.1066, 0.1058, 0.1049, 0.1041, 0.1033, 0.1026, 0.1019,
          0.1011, 0.1001, 0.0991, 0.0981, 0.0972, 0.0962, 0.0953, 0.0943, 0.0934, 0.0926, 0.0920,
          0.0917, 0.0914, 0.0910, 0.0907, 0.0902, 0.0898, 0.0894, 0.0891, 0.0887, 0.0884, 0.0879,
          0.0873, 0.0866, 0.0859, 0.0853, 0.0848, 0.0844, 0.0839, 0.0832, 0.0822, 0.0810, 0.0798,
          0.0788, 0.0779, 0.0772, 0.0765, 0.0759, 0.0753, 0.0749, 0.0746, 0.0746, 0.0746, 0.0745,
          0.0744, 0.0742, 0.0741, 0.0743, 0.0746, 0.0750, 0.0752, 0.0751, 0.0748, 0.0744, 0.0740,
          0.0738, 0.0738, 0.0740, 0.0744, 0.0747, 0.0750, 0.0752, 0.0754, 0.0754, 0.0754, 0.0752,
          0.0750, 0.0746, 0.0743, 0.0740, 0.0737, 0.0735, 0.0734, 0.0735, 0.0736, 0.0739, 0.0742,
          0.0744, 0.0745, 0.0747, 0.0751, 0.0757, 0.0764, 0.0770, 0.0771, 0.0768, 0.0762, 0.0755,
          0.0751, 0.0749, 0.0748, 0.0748, 0.0748, 0.0748, 0.0750, 0.0752, 0.0754, 0.0754, 0.0752,
          0.0748, 0.0744, 0.0741, 0.0739, 0.0737, 0.0736, 0.0736, 0.0739, 0.0743, 0.0750, 0.0755,
          0.0759, 0.0760, 0.0760, 0.0759, 0.0760, 0.0762, 0.0765, 0.0768, 0.0769, 0.0769, 0.0767,
          0.0766, 0.0765, 0.0766, 0.0767, 0.0767, 0.0765, 0.0761, 0.0757, 0.0755, 0.0756, 0.0760,
          0.0763, 0.0765, 0.0764, 0.0762, 0.0761, 0.0762, 0.0766, 0.0770, 0.0773, 0.0776, 0.0778,
          0.0779, 0.0781, 0.0782, 0.0781, 0.0778, 0.0776, 0.0774, 0.0774, 0.0776, 0.0778, 0.0781,
          0.0784, 0.0788, 0.0792, 0.0797, 0.0801, 0.0805, 0.0808, 0.0809, 0.0810, 0.0810, 0.0810,
          0.0809, 0.0808, 0.0808, 0.0808, 0.0807, 0.0807, 0.0807, 0.0808, 0.0809, 0.0810, 0.0811,
          0.0809, 0.0807, 0.0805, 0.0804, 0.0805, 0.0807, 0.0810, 0.0815, 0.0820, 0.0826, 0.0832,
          0.0837, 0.0840, 0.0841, 0.0841, 0.0842, 0.0844, 0.0845, 0.0845, 0.0842, 0.0839, 0.0837,
          0.0838, 0.0841, 0.0846, 0.0850, 0.0852, 0.0852, 0.0850, 0.0847, 0.0845, 0.0846, 0.0848,
          0.0853, 0.0860, 0.0865, 0.0868, 0.0869, 0.0867, 0.0864, 0.0861, 0.0859, 0.0858, 0.0858,
          0.0859, 0.0860, 0.0861, 0.0863, 0.0864, 0.0864, 0.0865, 0.0866, 0.0868, 0.0870, 0.0871,
          0.0872, 0.0872, 0.0870, 0.0869, 0.0869, 0.0871, 0.0875, 0.0881, 0.0887, 0.0893, 0.0896,
          0.0898, 0.0899, 0.0899, 0.0899, 0.0900, 0.0900, 0.0902, 0.0905, 0.0909, 0.0914, 0.0918,
          0.0919, 0.0919, 0.0919, 0.0920, 0.0924, 0.0930, 0.0935, 0.0937, 0.0936, 0.0933, 0.0930,
          0.0927, 0.0926, 0.0925, 0.0924, 0.0923, 0.0921, 0.0918, 0.0917, 0.0916, 0.0916, 0.0918,
          0.0919, 0.0920, 0.0919, 0.0918, 0.0918, 0.0919, 0.0921, 0.0923, 0.0926, 0.0930, 0.0934,
          0.0938, 0.0943, 0.0947, 0.0948, 0.0948, 0.0947, 0.0945, 0.0944, 0.0942, 0.0938, 0.0933,
          0.0928, 0.0923, 0.0918, 0.0914, 0.0908, 0.0902, 0.0898, 0.0896, 0.0895, 0.0895, 0.0893,
          0.0890, 0.0886, 0.0883, 0.0881, 0.0879, 0.0876, 0.0873, 0.0872, 0.0876, 0.0884, 0.0895,
          0.0904, 0.0908, 0.0908, 0.0904, 0.0901, 0.0898, 0.0897, 0.0895, 0.0892, 0.0887, 0.0882,
          0.0877, 0.0871, 0.0866, 0.0861, 0.0857, 0.0855, 0.0853, 0.0853, 0.0853, 0.0854, 0.0856,
          0.0859, 0.0861, 0.0861, 0.0859, 0.0856, 0.0853, 0.0851, 0.0849, 0.0848, 0.0848, 0.0848,
          0.0850, 0.0852, 0.0853, 0.0852, 0.0850, 0.0848, 0.0849, 0.0853, 0.0857, 0.0860, 0.0861,
          0.0859, 0.0857, 0.0855, 0.0852, 0.0849, 0.0844, 0.0838, 0.0833, 0.0829, 0.0825, 0.0821,
          0.0819, 0.0819, 0.0822, 0.0828, 0.0832, 0.0834, 0.0833, 0.0829, 0.0825, 0.0822, 0.0820,
          0.0818, 0.0816, 0.0815, 0.0815, 0.0815, 0.0814, 0.0812, 0.0808, 0.0805, 0.0803, 0.0803,
          0.0802, 0.0799, 0.0794, 0.0788, 0.0785, 0.0785, 0.0785, 0.0785, 0.0783, 0.0780, 0.0778,
          0.0776, 0.0774, 0.0769, 0.0764, 0.0758, 0.0755, 0.0754, 0.0754, 0.0753, 0.0749, 0.0745,
          0.0742, 0.0741, 0.0741, 0.0740, 0.0739, 0.0738, 0.0739, 0.0740, 0.0740, 0.0737, 0.0732,
          0.0729, 0.0728, 0.0729, 0.0730, 0.0729, 0.0725, 0.0724, 0.0727, 0.0733, 0.0739, 0.0740,
          0.0735, 0.0728, 0.0723, 0.0723, 0.0725, 0.0725, 0.0723, 0.0720, 0.0720, 0.0722, 0.0724,
          0.0722, 0.0716, 0.0711, 0.0710, 0.0713, 0.0718, 0.0719, 0.0716, 0.0714, 0.0716, 0.0724,
          0.0730, 0.0731, 0.0725, 0.0718, 0.0717, 0.0723, 0.0731, 0.0734, 0.0730, 0.0723, 0.0722,
          0.0728, 0.0735, 0.0735, 0.0725, 0.0713, 0.0711, 0.0722, 0.0737, 0.0740, 0.0722, 0.0692,
          0.0669}
{
    voiceStats->calcOnVqe(config.useVqeChannel);
}

void SmartVolumeManager::RMSSource::onAudioData(const ChannelsData& data)
{
    voiceStats->pushAudioChannels(data);
}

quasar::RMSInfo SmartVolumeManager::RMSSource::getRms() const {
    YIO_LOG_INFO("Use RMS weights: " << useRmsWeights);
    if (useRmsWeights) {
        return voiceStats->getWeightedRms(rmsWeights);
    } else {
        return voiceStats->getRms();
    }
}

SmartVolumeManager::SmartVolumeManager(std::shared_ptr<VolumeManager> volumeManager, std::shared_ptr<IAudioSourceClient> audioSource, Config config)
    : volumeManager_{std::move(volumeManager)}
    , rmsSource_{std::make_shared<RMSSource>(config)}
    , config_{std::move(config)}
{
    Y_VERIFY(volumeManager_);
    Y_VERIFY(!config_.rmsVolumeMap.empty());

    YIO_LOG_INFO("Use config. " << config_);

    audioSource->addListener(rmsSource_);
}

void SmartVolumeManager::updateConfig(SmartVolumeManager::Config config)
{
    Y_VERIFY(!config.rmsVolumeMap.empty());

    YIO_LOG_INFO("Update config. " << config);

    config_ = std::move(config);
    rmsSource_->useRmsWeights = config_.useVoiceFilter;
    rmsSource_->voiceStats->calcOnVqe(config_.useVqeChannel);
}

int SmartVolumeManager::getVolumeByRms(int rms) const {
    int volume = config_.rmsVolumeMap.rbegin()->second;

    for (const auto [rmsValue, volumeValue] : config_.rmsVolumeMap) {
        if (rms < rmsValue) {
            volume = volumeValue - 1;
            break;
        }
    }

    return volume;
}

void SmartVolumeManager::onAliceStateChanged(proto::AliceState state)
{
    switch (state.state()) {
        case proto::AliceState::BUSY: {
            if (restoreVolumeMode_) {
                return;
            }

            if (state.has_vins_response()) {
                if (const auto& vinsResponse = state.vins_response(); vinsResponse.has_output_speech()) {
                    const auto rmsInfo = rmsSource_->getRms();
                    const auto jsonRms = VoiceStats::rmsToJsonPacked(rmsInfo);

                    const int rmsValue = jsonRms["RawAvgRMS"].asInt();
                    const int volumeValue = getVolumeByRms(rmsValue);

                    YIO_LOG_INFO("RMS value: " << rmsValue << ", Alice volume: " << volumeValue);

                    restoreVolumeMode_ = true;

                    volumeManager_->enableStashVolumeMode(volumeManager_->scaleFromAlice(volumeValue));
                }
            }
        }
        case proto::AliceState::SPEAKING:
        case proto::AliceState::IDLE:
        default:
            break;
    }
}

void SmartVolumeManager::onAliceTtsCompleted()
{
    if (restoreVolumeMode_) {
        restoreVolumeMode_ = false;
        volumeManager_->disableStashVolumeMode();
    }
}
