#include "spotter_configurer.h"
#include "spotter_channels.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <voicetech/library/channel_types/string_names/channel_string_names.h>

#include <set>
#include <sstream>

YIO_DEFINE_LOG_MODULE("spotter_configurer");

using namespace YandexIO;

namespace {

    constexpr auto SPOTTER_CONFIG_NAME = "spotter";
    constexpr auto ENABLE_SPOTTERS_CONFIG_NAME = "enableSpotters";

    const std::set<std::string> SYSTEM_CONFIG_NAMES = {SPOTTER_CONFIG_NAME, ENABLE_SPOTTERS_CONFIG_NAME};

    std::string channelMapToString(const std::map<AudioInputChannelType, uint8_t>& channels) {
        std::stringstream ss;
        for (const auto& [type, count] : channels) {
            ss << "[" << AudioInputChannelType_ToString(type) << ":" << static_cast<int>(count) << "]";
        }
        return ss.str();
    }

    std::string modelsToString(const std::map<std::string, std::string>& typeToModels) {
        std::stringstream ss;
        for (const auto& [type, path] : typeToModels) {
            ss << type << "=";
            if (!path.empty()) {
                ss << path << "=" << channelMapToString(getRequiredChannelsFromSpotterPath(path)) << " ";
            } else {
                ss << "empty ";
            }
        }
        return ss.str();
    }

    constexpr auto DEFAULT_SPOTTER_WORD = "alisa";

} // namespace

std::shared_ptr<SpotterConfigurer> SpotterConfigurer::install(
    std::unique_ptr<quasar::ICallbackQueue> callbackQueue,
    SDKInterface& sdk,
    std::shared_ptr<IDevice> device)
{
    auto configurer = std::shared_ptr<SpotterConfigurer>(new SpotterConfigurer(std::move(callbackQueue), sdk, std::move(device)));
    configurer->init(sdk);
    return configurer;
}

SpotterConfigurer::~SpotterConfigurer() {
    spotterDownloader_.stop();

    activationSpotter_->removeListener(weak_from_this());
    commandSpotter_->removeListener(weak_from_this());
    naviOldSpotter_->removeListener(weak_from_this());

    callbackQueue_.reset();
}

void SpotterConfigurer::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (!SYSTEM_CONFIG_NAMES.contains(configName)) {
        return;
    }

    Json::Value jsonConfig;
    try {
        jsonConfig = quasar::parseJson(jsonConfigValue);
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("SpotterConfigurer.InvalidSystemConfigJson", e.what());
        return;
    }

    if (configName == SPOTTER_CONFIG_NAME) {
        processSpotterConfig(jsonConfig);
    } else if (configName == ENABLE_SPOTTERS_CONFIG_NAME) {
        processEnableSpottersConfig(jsonConfig);
    }
}

void SpotterConfigurer::onAccountConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != SPOTTER_CONFIG_NAME) {
        return;
    }

    std::string newSpotterWord = DEFAULT_SPOTTER_WORD;
    try {
        const auto jsonConfig = quasar::parseJson(jsonConfigValue);
        if (jsonConfig.isString()) {
            newSpotterWord = jsonConfig.asString();
        }
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("SpotterConfigurer.InvalidAccountConfigJson", e.what());
        return;
    }

    callbackQueue_->add([this, newSpotterWord]() mutable {
        if (newSpotterWord == spotterWord_) {
            return;
        }

        spotterWord_ = std::move(newSpotterWord);

        YIO_LOG_INFO("New spotter word: " << spotterWord_);

        provideSpotterWord();

        updateSpotterDownloaderState();
    });
}

void SpotterConfigurer::onModelError(std::shared_ptr<ISpotterCapability> /*capability*/, const std::set<std::string>& spotterTypes) {
    callbackQueue_->add([this, spotterTypes] {
        for (const auto& spotterType : spotterTypes) {
            YIO_LOG_INFO("Model error with spotter of type [" << spotterType << "]");

            spotterDownloader_.uninstallSpotter(spotterType);

            processSpotterModelUpdate(spotterType);
        }
    });
}

void SpotterConfigurer::onModelSet(std::shared_ptr<ISpotterCapability> /*capability*/, const std::set<std::string>& spotterTypes) {
    callbackQueue_->add([this, spotterTypes] {
        spotterDownloader_.emptyTrash(spotterTypes);
    });
}

SpotterDownloader& SpotterConfigurer::getSpotterDownloader() {
    return spotterDownloader_;
}

SpotterConfigurer::SpotterConfigurer(
    std::unique_ptr<quasar::ICallbackQueue> callbackQueue,
    SDKInterface& sdk,
    std::shared_ptr<IDevice> device)
    : callbackQueue_(std::move(callbackQueue))
    , device_(std::move(device))
    , spotterDownloader_(device_, [this](const SpotterDownloader::SpotterUrlInfo& spotterInfo) {
        callbackQueue_->add([this, spotterInfo]() {
            processSpotterModelUpdate(spotterInfo.type);
        });
    })
    , defaultEnableSpotters_(quasar::tryGetBool(device_->configuration()->getServiceConfig("aliced"), "enableSpotters", true))
    , enableSpotters_(defaultEnableSpotters_)
    , spotterWord_(DEFAULT_SPOTTER_WORD)
    , activationSpotter_(sdk.getActivationSpotterCapability())
    , commandSpotter_(sdk.getCommandSpotterCapability())
    , naviOldSpotter_(sdk.getNaviOldSpotterCapability())
{
    spotterDownloader_.start();
}

void SpotterConfigurer::init(SDKInterface& sdk) {
    for (const auto& config : SYSTEM_CONFIG_NAMES) {
        sdk.subscribeToSystemConfig(config);
    }
    sdk.subscribeToAccountConfig(SPOTTER_CONFIG_NAME);
    sdk.addBackendConfigObserver(weak_from_this());

    activationSpotter_->addListener(weak_from_this());
    commandSpotter_->addListener(weak_from_this());
    naviOldSpotter_->addListener(weak_from_this());

    callbackQueue_->add([this] {
        updateSpotters();
        provideSpotterWord();
    });
}

void SpotterConfigurer::processSpotterConfig(const Json::Value& jsonConfig) {
    callbackQueue_->add([this, newSpotterConfig{jsonConfig}]() mutable {
        if (newSpotterConfig == spotterConfig_) {
            return;
        }

        spotterConfig_ = std::move(newSpotterConfig);

        updateSpotterDownloaderState();
    });
}

void SpotterConfigurer::processEnableSpottersConfig(const Json::Value& jsonConfig) {
    if (!jsonConfig.isBool() && !jsonConfig.isNull()) {
        return;
    }

    callbackQueue_->add([this, jsonConfig] {
        auto newEnableSpotters = jsonConfig.isBool() ? jsonConfig.asBool() : defaultEnableSpotters_;
        if (enableSpotters_ == newEnableSpotters) {
            return;
        }

        enableSpotters_ = newEnableSpotters;

        // TODO:
        // activationSpotter_->setEnabled() & commandSpotter_->setEnabled() & naviOldSpotter_->setEnabled()
        // remove handling from SKE
    });
}

void SpotterConfigurer::processSpotterModelUpdate(const std::string& spotterType) {
    callbackQueue_->add([this, spotterType] {
        if (spotterType == quasar::SpotterTypes::NAVIGATION_OLD) {
            updateNaviOldSpotters(getNaviOldModels());
        } else if (spotterType == quasar::SpotterTypes::ACTIVATION ||
                   spotterType == quasar::SpotterTypes::INTERRUPTION ||
                   spotterType == quasar::SpotterTypes::ADDITIONAL)
        {
            updateActivationSpotters(getActivationModels());
        } else {
            updateCommandSpotters(getCommandModels());
        }
    });
}

void SpotterConfigurer::updateSpotterDownloaderState() {
    if (!enableSpotters_) {
        YIO_LOG_INFO("Skip spotters downloading due to enableSpotters: false.");
        return;
    }

    std::map<std::string, SpotterDownloader::SpotterUrlInfo> spotters;
    for (const auto& key : spotterConfig_.getMemberNames()) {
        const Json::Value& entry = spotterConfig_[key];
        YIO_LOG_DEBUG(quasar::jsonToString(entry));
        SpotterDownloader::SpotterUrlInfo spotter;
        if (!entry.isMember("type")) {
            // in case of missing type fill values for backward compatibility
            spotter.type = quasar::SpotterTypes::ACTIVATION;
            spotter.word = key;
        } else {
            spotter.type = quasar::tryGetString(entry, "type", quasar::SpotterTypes::ACTIVATION);
            spotter.word = quasar::tryGetString(entry, "word", "");
        }
        // For activation spotter download only words that are required at the current moment.
        // Don't download alice spotter if yandex is set in user config
        if (spotter.word.empty() || spotterWord_ == spotter.word) {
            spotter.url = quasar::tryGetString(entry, "url");
            spotter.crc32 = quasar::tryGetUInt64(entry, "crc", 0);
            spotter.archFormat = quasar::tryGetString(entry, "format", "");
            spotters[spotter.type] = spotter;
        }
    }
    spotterDownloader_.changeSpotterUrls(std::move(spotters));

    updateSpotters();
}

void SpotterConfigurer::provideSpotterWord() {
    activationSpotter_->setSpotterWord(spotterWord_);
    device_->telemetry()->putAppEnvironmentValue("spotterModel", spotterWord_);
}

std::map<std::string, std::string> SpotterConfigurer::getActivationModels() const {
    return {
        {quasar::SpotterTypes::ACTIVATION, spotterDownloader_.getSpotterPath(quasar::SpotterTypes::ACTIVATION, spotterWord_)},
        {quasar::SpotterTypes::INTERRUPTION, spotterDownloader_.getSpotterPath(quasar::SpotterTypes::INTERRUPTION, spotterWord_)},
        {quasar::SpotterTypes::ADDITIONAL, spotterDownloader_.getSpotterPath(quasar::SpotterTypes::ADDITIONAL, "")},
    };
}

std::map<std::string, std::string> SpotterConfigurer::getNaviOldModels() const {
    return {{quasar::SpotterTypes::NAVIGATION_OLD, spotterDownloader_.getSpotterPath(quasar::SpotterTypes::NAVIGATION_OLD, "")}};
}

std::map<std::string, std::string> SpotterConfigurer::getCommandModels() const {
    std::map<std::string, std::string> modelPaths;
    for (const auto& spotterType : quasar::SpotterTypes::COMMAND_ALL) {
        modelPaths[spotterType] = spotterDownloader_.getSpotterPath(spotterType, "");
    }
    return modelPaths;
}

void SpotterConfigurer::updateSpotters() {
    if (auto newSpotters = getActivationModels(); newSpotters != activationModels_) {
        updateActivationSpotters(std::move(newSpotters));
    }

    if (auto newSpotters = getCommandModels(); newSpotters != commandModels_) {
        updateCommandSpotters(std::move(newSpotters));
    }

    if (auto newSpotters = getNaviOldModels(); newSpotters != naviOldModels_) {
        updateNaviOldSpotters(std::move(newSpotters));
    }
}

void SpotterConfigurer::updateActivationSpotters(std::map<std::string, std::string> typeToModels) {
    YIO_LOG_INFO("Update activation spotters: " << modelsToString(typeToModels));

    activationModels_ = std::move(typeToModels);

    activationSpotter_->setModelPaths(activationModels_);
}

void SpotterConfigurer::updateCommandSpotters(std::map<std::string, std::string> typeToModels) {
    YIO_LOG_INFO("Update command spotters: " << modelsToString(typeToModels));

    commandModels_ = std::move(typeToModels);

    commandSpotter_->setModelPaths(commandModels_);
}

void SpotterConfigurer::updateNaviOldSpotters(std::map<std::string, std::string> typeToModels) {
    YIO_LOG_INFO("Update naviOld spotters: " << modelsToString(typeToModels));

    naviOldModels_ = std::move(typeToModels);

    naviOldSpotter_->setModelPaths(naviOldModels_);
}
