#include "spotter_downloader.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>
#include <yandex_io/libs/zip/mini_unzip.h>

#include <util/generic/scope.h>

YIO_DEFINE_LOG_MODULE("spotter_downloader");

using namespace YandexIO;
using namespace quasar;

namespace {

    const std::string SERVICE_NAME = "aliced";

    void removeDir(const std::string& path) {
        TFsPath spotterDir(path);
        if (spotterDir.Exists()) {
            spotterDir.ForceDelete();
        } else {
            YIO_LOG_WARN("Cleanup directory is missing. Path: " << spotterDir.GetPath());
        }
    }

    Json::Value prepareMetricaEvent(const std::string& spotterUrl, const std::string& spotterWord) {
        Json::Value event;
        event["url"] = spotterUrl;
        event["word"] = spotterWord;

        return event;
    }

} // unnamed namespace

SpotterDownloader::SpotterDownloader(std::shared_ptr<IDevice> device, std::function<void(const SpotterUrlInfo& spotterInfo)> onSpotterDownloaded)
    : device_(std::move(device))
    , httpClient_("mds", device_)
    , onSpotterDownloaded_(std::move(onSpotterDownloaded))
{
    const auto& aliceConfig_ = device_->configuration()->getServiceConfig(SERVICE_NAME);
    customSpotterConfigPath_ = TFsPath(tryGetString(aliceConfig_, "customSpotterConfigPath"));
    customSpotterDir_ = TFsPath(tryGetString(aliceConfig_, "customSpotterDir"));

    initialRetryTimeoutMs_ = tryGetInt(aliceConfig_, "downloaderInitialRetryTimeoutMs", 1000);
    maxRetryTimeoutMs_ = tryGetInt(aliceConfig_, "downloaderMaxRetryTimeoutMs", 60000);

    const auto& commonConfig = device_->configuration()->getServiceConfig("common");
    tempDir_ = TFsPath(tryGetString(commonConfig, "tempDir"));

    Y_VERIFY(customSpotterConfigPath_.IsDefined());
    Y_VERIFY(customSpotterDir_.IsDefined());
    Y_VERIFY(tempDir_.IsDefined());

    httpClient_.setTimeout(std::chrono::milliseconds{50000});
}

SpotterDownloader::~SpotterDownloader() {
    stop();
}

void SpotterDownloader::start() {
    if (customSpotterConfigPath_.Exists()) {
        loadConfig();
    }

    if (!tempDir_.Exists()) {
        tempDir_.MkDirs();
    }
    if (!customSpotterDir_.Exists()) {
        customSpotterDir_.MkDirs();
    }
}

void SpotterDownloader::stop() {
    std::lock_guard<std::mutex> lock(mutex_);

    if (stopped_.load()) {
        return;
    }

    httpClient_.cancelDownload();
    stopped_ = true;
    breakThreadCond_.notify_one();
}

std::string SpotterDownloader::getSpotterPath(const std::string& spotterType, const std::string& spotterWord) const {
    std::lock_guard<std::mutex> lock(mutex_);

    if (installedSpotters_.find(spotterType) != installedSpotters_.end()) {
        auto& entry = installedSpotters_.at(spotterType);
        if (TFsPath(entry.path).Exists() && SpotterValidator::isValid(entry.path, entry.crc32) && entry.word == spotterWord) {
            YIO_LOG_DEBUG("Type [" << spotterType << "] word [" << spotterWord << "]: custom path: " << entry.path);
            return entry.path;
        }
    }

    std::string defaultSpotterPath = getDefaultSpotterPath(spotterType, spotterWord);

    if (fileExists(defaultSpotterPath)) {
        YIO_LOG_DEBUG("Type [" << spotterType << "] word [" << spotterWord << "]: default path: " << defaultSpotterPath);
        return defaultSpotterPath;
    }

    if (!spotterWord.empty() && spotterWord != "alisa") {
        // Try to fallback to default spotter word 'alisa'
        const std::string oldSpotterPath = defaultSpotterPath;
        defaultSpotterPath = getDefaultSpotterPath(spotterType, "alisa");
        if (fileExists(defaultSpotterPath)) {
            YIO_LOG_DEBUG("Type [" << spotterType << "] word [" << spotterWord << "]: fallback alisa path: " << defaultSpotterPath);
            return defaultSpotterPath;
        }
    }

    YIO_LOG_DEBUG("Type [" << spotterType << "] word [" << spotterWord << "]: empty path");
    return {};
}

SpotterDownloader::ArchiveFormat SpotterDownloader::archFormatFromString(const std::string& format) {
    if (format.empty() || format == "targz") {
        return ArchiveFormat::TAR_GZ;
    } else if (format == "zip") {
        return ArchiveFormat::ZIP;
    } else {
        throw std::runtime_error("Unknown archive format " + format);
    }
}

void SpotterDownloader::changeSpotterUrls(std::map<std::string, SpotterUrlInfo> newSpotters) {
    YIO_LOG_DEBUG("changeSpotterUrls called");

    std::lock_guard<std::mutex> lock(mutex_);

    checkDownloading(newSpotters);
    checkPending(newSpotters);
    checkInstalled(newSpotters);

    saveConfig();

    for (const auto& [type, spotterInfo] : newSpotters) {
        downloadingSpotters_[type] = spotterInfo;
        asyncQueue_.add([this, spotterInfo = spotterInfo]() {
            getNewSpotter(spotterInfo);
        });
    }
}

uint32_t SpotterDownloader::downloadingTasksCount() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return downloadingSpotters_.size();
}

void SpotterDownloader::uninstallSpotter(const std::string& type) {
    std::lock_guard<std::mutex> lock(mutex_);

    if (auto it = installedSpotters_.find(type); it != installedSpotters_.end()) {
        YIO_LOG_DEBUG("Removing spotter of type " << type);
        removeDir(it->second.path);
        installedSpotters_.erase(it);
        saveConfig();
    }
}

void SpotterDownloader::emptyTrash(const std::set<std::string>& spotterTypes) {
    std::lock_guard<std::mutex> lock(mutex_);

    for (const auto& type : spotterTypes) {
        if (auto it = trashSpotterPaths_.find(type); it != trashSpotterPaths_.end()) {
            YIO_LOG_DEBUG("Removing spotter of type " << type << " from trash");
            removeDir(it->second);
            trashSpotterPaths_.erase(it);
        }
    }
}

void SpotterDownloader::getNewSpotter(const SpotterUrlInfo& spotterUrl) {
    std::unique_lock<std::mutex> lock(mutex_);

    // directory where downloaded archive is stored
    TFsPath tempDownloadDir(JoinFsPaths(tempDir_, makeUUID()));
    tempDownloadDir.MkDirs();
    YIO_LOG_INFO("Getting new spotter from " << spotterUrl.url);
    // directory with unverified spotter. Must be on data partition
    customSpotterDir_.MkDirs();
    TFsPath spotterPath(JoinFsPaths(customSpotterDir_, makeUUID()));

    auto isDownloadStillActual = [&]() {
        auto it = downloadingSpotters_.find(spotterUrl.type);
        return it != downloadingSpotters_.end() && it->second == spotterUrl;
    };

    bool success = false;
    Y_DEFER {
        YIO_LOG_DEBUG("removing tempDownloadDir at " << tempDownloadDir);
        tempDownloadDir.ForceDelete();
        if (!success) {
            YIO_LOG_DEBUG("removing spotterPath at " << spotterPath);
            spotterPath.ForceDelete();
        }
    };

    uint32_t timeoutMs = initialRetryTimeoutMs_;
    const std::string spotterName = getFileName(spotterUrl.url);
    const std::string downloadedArchivePath = JoinFsPaths(tempDownloadDir, spotterName);
    Json::Value metricaEvent = prepareMetricaEvent(spotterUrl.url, spotterUrl.word);

    while (!stopped_.load() && isDownloadStillActual()) {
        lock.unlock();
        bool downloaded = downloadSpotterArchive(spotterUrl.url, downloadedArchivePath, metricaEvent);
        lock.lock();

        if (downloaded && isDownloadStillActual()) {
            if (!extractSpotter(downloadedArchivePath, spotterPath.GetPath(), metricaEvent, archFormatFromString(spotterUrl.archFormat))) {
                device_->telemetry()->reportEvent("customSpotterExtractFail", jsonToString(metricaEvent));
                downloadingSpotters_.erase(spotterUrl.type);
                return;
            }

            if (!SpotterValidator::isValid(spotterPath.GetPath(), spotterUrl.crc32)) {
                device_->telemetry()->reportEvent("customSpotterValidationFail", jsonToString(metricaEvent));
                downloadingSpotters_.erase(spotterUrl.type);
                return;
            }

            // destination directory
            installedSpotters_[spotterUrl.type] = spotterUrl;
            installedSpotters_[spotterUrl.type].path = spotterPath.GetPath();
            downloadingSpotters_.erase(spotterUrl.type);
            success = true;
            saveConfig();
            if (onSpotterDownloaded_) {
                onSpotterDownloaded_(spotterUrl);
            }
            return;
        }
        breakThreadCond_.wait_for(lock, std::chrono::milliseconds(timeoutMs), [this]() {
            return stopped_.load();
        });
        timeoutMs = std::min(timeoutMs * 2, maxRetryTimeoutMs_);
    }
    if (isDownloadStillActual()) {
        downloadingSpotters_.erase(spotterUrl.type);
    }
}

void SpotterDownloader::loadConfig() {
    try {
        const Json::Value customSpotterConfig = getConfigFromFile(customSpotterConfigPath_.GetPath());
        YIO_LOG_DEBUG("Load config: " << jsonToString(customSpotterConfig));
        for (const auto& key : customSpotterConfig.getMemberNames()) {
            const auto& entry = customSpotterConfig[key];
            SpotterUrlInfo configEntry;
            configEntry.type = key;
            configEntry.word = entry["word"].asString();
            configEntry.path = entry["path"].asString();
            configEntry.url = entry["url"].asString();
            configEntry.crc32 = entry["crc"].asUInt();
            TFsPath path(configEntry.path);
            if (path.Exists() && path.IsDirectory()) {
                pendingSpotters_.emplace(key, configEntry);
            }
        }

        // cleanup
        TVector<TFsPath> danglingSpotters;
        customSpotterDir_.List(danglingSpotters);
        for (auto& item : danglingSpotters) {
            auto pendingIt = std::find_if(pendingSpotters_.begin(), pendingSpotters_.end(), [&item](const auto& pending) {
                return pending.second.path == item;
            });
            if (pendingIt == pendingSpotters_.end()) {
                YIO_LOG_DEBUG("Removing dangling spotter " << item);
                item.ForceDelete();
            }
        }
    } catch (const std::exception& e) {
        // remove invalid config and clean up spotter directory to avoid zombie spotters in it
        customSpotterConfigPath_.ForceDelete();
        customSpotterDir_.ForceDelete();
        pendingSpotters_.clear();
        YIO_LOG_ERROR_EVENT("SpotterDownloader.BadJson.SpotterConfig", "Failed to parse spotter.json " << e.what())
    }
}

void SpotterDownloader::saveConfig() {
    Json::Value config;
    for (const auto& kv : installedSpotters_) {
        auto& entry = config[kv.first];
        entry["word"] = kv.second.word;
        entry["path"] = kv.second.path;
        entry["crc"] = kv.second.crc32;
        entry["url"] = kv.second.url;
    }
    PersistentFile file(customSpotterConfigPath_.GetPath(), PersistentFile::Mode::TRUNCATE);
    file.write(jsonToString(config));
}

void SpotterDownloader::checkDownloading(std::map<std::string, SpotterUrlInfo>& newSpotters) {
    for (auto downloadingIt = downloadingSpotters_.begin(); downloadingIt != downloadingSpotters_.end();) {
        if (auto it = newSpotters.find(downloadingIt->first); it != newSpotters.end() && it->second == downloadingIt->second) {
            // the same spotter is already downdloading
            newSpotters.erase(it);
            ++downloadingIt;
        } else {
            downloadingIt = downloadingSpotters_.erase(downloadingIt);
        }
    }
}

void SpotterDownloader::checkPending(std::map<std::string, SpotterUrlInfo>& newSpotters) {
    for (auto pendingIt = pendingSpotters_.begin(); pendingIt != pendingSpotters_.end(); ++pendingIt) {
        if (auto it = newSpotters.find(pendingIt->first); it != newSpotters.end() && it->second == pendingIt->second) {
            // pending spotter has waited for its config
            installedSpotters_[pendingIt->first] = std::move(pendingIt->second);
        } else {
            removeDir(pendingIt->second.path);
        }
    }
    pendingSpotters_.clear();
}

void SpotterDownloader::checkInstalled(std::map<std::string, SpotterUrlInfo>& newSpotters) {
    for (auto installedIt = installedSpotters_.begin(); installedIt != installedSpotters_.end();) {
        if (auto it = newSpotters.find(installedIt->first); it != newSpotters.end() && it->second == installedIt->second) {
            // the same spotter is already installed
            newSpotters.erase(it);
            ++installedIt;
        } else {
            // move to trash unused spotter
            trashSpotterPaths_.emplace(installedIt->first, installedIt->second.path);
            installedIt = installedSpotters_.erase(installedIt);
        }
    }
}

bool SpotterDownloader::extractSpotter(const std::string& archivePath, const TFsPath& destPath, Json::Value& metricaEventBody, ArchiveFormat format) {
    try {
        if (!destPath.Exists()) {
            destPath.MkDirs();
        }
        switch (format) {
            case ArchiveFormat::TAR_GZ:
                extractTargzArchive(archivePath, destPath.GetPath());
                break;
            case ArchiveFormat::ZIP:
                extractZipArchive(archivePath, destPath.GetPath());
                break;
        }
        YIO_LOG_INFO("Spotter archive successfully extracted at: " + destPath.GetPath());
        return true;
    } catch (const std::exception& e) {
        metricaEventBody["error"] = e.what();
        return false;
    }
}

bool SpotterDownloader::downloadSpotterArchive(const std::string& url, const std::string& path, Json::Value& metricaEventBody) {
    try {
        TFsPath(path).ForceDelete();
        httpClient_.download("spotter", url, path);
        YIO_LOG_INFO("Spotter successfully downloaded to " + path);
        device_->telemetry()->reportEvent("customSpotterDownloaded", jsonToString(metricaEventBody));
        return true;
    } catch (const std::exception& e) {
        metricaEventBody["error"] = e.what();
        device_->telemetry()->reportEvent("customSpotterDownloadFail", jsonToString(metricaEventBody));
        YIO_LOG_INFO("Failed to download spotter url " << url << " because of " << e.what());
        return false;
    }
}

std::string SpotterDownloader::getDefaultSpotterPath(const std::string& spotterType, const std::string& spotterWord) const {
    const auto aliceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
    const auto modelsPath = getString(aliceConfig, "spotterModelsPath");
    std::string result = modelsPath + "/" + spotterType;
    if (!spotterWord.empty()) {
        result += "/" + spotterWord;
    }
    return result;
}
