#pragma once

#include "spotter_validator.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <util/folder/path.h>

#include <string>
#include <thread>

namespace YandexIO {

    class SpotterDownloader {
    public:
        struct SpotterUrlInfo {
            std::string url;
            std::string type;
            std::string word;
            uint32_t crc32;

            std::string path;
            std::string archFormat;

            bool operator==(const SpotterUrlInfo& other) const {
                return url == other.url && type == other.type && word == other.word && crc32 == other.crc32;
            }

            bool operator!=(const SpotterUrlInfo& other) const {
                return !(*this == other);
            }
        };

    public:
        SpotterDownloader(std::shared_ptr<IDevice> device, std::function<void(const SpotterUrlInfo& spotterInfo)> onSpotterDownloaded);
        ~SpotterDownloader();

        void start();
        void stop();

        std::string getSpotterPath(const std::string& spotterType, const std::string& spotterWord) const;
        void changeSpotterUrls(std::map<std::string, SpotterUrlInfo> newSpotters);
        void uninstallSpotter(const std::string& spotterType);
        void emptyTrash(const std::set<std::string>& spotterTypes);

        // for testing purposes
        uint32_t downloadingTasksCount() const;

    private:
        enum class ArchiveFormat {
            TAR_GZ,
            ZIP
        };
        static ArchiveFormat archFormatFromString(const std::string& format);
        static bool extractSpotter(const std::string& archivePath, const TFsPath& destPath, Json::Value& metricaEventBody, ArchiveFormat format);

        void loadConfig();
        void saveConfig();

        void checkDownloading(std::map<std::string, SpotterUrlInfo>& newSpotters);
        void checkPending(std::map<std::string, SpotterUrlInfo>& newSpotters);
        void checkInstalled(std::map<std::string, SpotterUrlInfo>& newSpotters);

        std::string getDefaultSpotterPath(const std::string& spotterType, const std::string& spotterWord) const;
        void getNewSpotter(const SpotterUrlInfo& spotterUrl);
        bool downloadSpotterArchive(const std::string& url, const std::string& path, Json::Value& metricaEventBody);

    private:
        quasar::NamedCallbackQueue asyncQueue_{"SpotterDownloader"};
        std::shared_ptr<IDevice> device_;
        quasar::HttpClient httpClient_;
        std::function<void(const SpotterUrlInfo& spotterInfo)> onSpotterDownloaded_;
        TFsPath customSpotterConfigPath_;
        TFsPath customSpotterDir_;
        TFsPath tempDir_;
        uint32_t initialRetryTimeoutMs_;
        uint32_t maxRetryTimeoutMs_;

        // Spotters in progress
        std::map<std::string, SpotterUrlInfo> downloadingSpotters_;
        // Installed spotters stored on disk
        std::map<std::string, SpotterUrlInfo> installedSpotters_;
        // Spotters loaded from disk pending config confirmation
        std::map<std::string, SpotterUrlInfo> pendingSpotters_;
        // Spotters waiting for approving to remove
        std::map<std::string, std::string> trashSpotterPaths_;

        quasar::SteadyConditionVariable breakThreadCond_;
        mutable std::mutex mutex_;
        std::atomic_bool stopped_{false};
    };

} // namespace YandexIO
