#include "remote_on_configure_done_greeting_observer.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <iostream>
#include <utility>

using namespace YandexIO;

RemoteOnConfigureDoneGreetingObserver::RemoteOnConfigureDoneGreetingObserver(
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    DeviceModeObserver::ConfigurationSuccessState configurationSuccessState,
    std::string soundName,
    const Settings& settings)
    : sdk_(std::move(sdk))
    , waitResponseDirectiveTimeout_(settings.waitResponseDirectiveTimeout)
    , configurationSuccessState_(configurationSuccessState)
    , allServicesConnectingTimeout_(settings.allServicesConnectingTimeout)
    , startingConfigureEnabled_(settings.enabled)
    , soundName_(std::move(soundName))
    , demoModeListener_(std::make_shared<DemoModeStatusListener>())
{
}

RemoteOnConfigureDoneGreetingObserver::~RemoteOnConfigureDoneGreetingObserver() {
    lifetime_.die();
    asyncQueue_.destroy();
}

void RemoteOnConfigureDoneGreetingObserver::onConfigureSuccessUpdate(DeviceModeObserver::ConfigurationSuccessState configurationSuccessState) {
    YIO_LOG_INFO("onConfigureSuccessUpdate");
    asyncQueue_.add([this, configurationSuccessState] {
        if (configurationSuccessState_ != configurationSuccessState) {
            return;
        }

        if (configReceived_ && !startingConfigureEnabled_) {
            onFallBack();
            lifetime_.die();
            return;
        }

        asyncQueue_.addDelayed([this] {
            if (!startingConfigureEnabled_) {
                onFallBack();
                return;
            }
            requestSemanticFrame();
        }, allServicesConnectingTimeout_, lifetime_);
    });
}

void RemoteOnConfigureDoneGreetingObserver::requestAction() {
}

void RemoteOnConfigureDoneGreetingObserver::requestSemanticFrame() {
    asyncQueue_.add([this] {
        isDirectiveWait_ = true;
        requestAction();
        asyncQueue_.addDelayed([this] {
            onFallBack();
        }, waitResponseDirectiveTimeout_, lifetime_);
    });
}

void RemoteOnConfigureDoneGreetingObserver::onDirective(const std::string& name, const std::string& /*vinsRequestId*/, const std::string& /*jsonPayload*/) {
    if (name == quasar::Directives::SUCCESS_STARTING_ONBOARDING && isDirectiveWait_) {
        YIO_LOG_INFO("onDirective");
        sdk_->getFilePlayerCapability()->stopSoundFile(soundName_);
        isDirectiveWait_ = false;
        lifetime_.die(); // stop onFallBack
    }
}

void RemoteOnConfigureDoneGreetingObserver::subscribeToSDK() {
    sdk_->addDeviceModeObserver(shared_from_this());
    sdk_->addBackendConfigObserver(shared_from_this());
    sdk_->addDirectiveObserver(shared_from_this());
}

void RemoteOnConfigureDoneGreetingObserver::onFallBack() {
    asyncQueue_.add([this] {
        sdk_->getAliceCapability()->stopConversation();
        if (!demoModeListener_->isInDemoMode()) {
            YIO_LOG_INFO("onFallBack");
            sdk_->getFilePlayerCapability()->playSoundFile(soundName_, quasar::proto::AudioChannel::DIALOG_CHANNEL);
        }
    });
}

void RemoteOnConfigureDoneGreetingObserver::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    if (configName != CONFIG_NAME) {
        return;
    }

    asyncQueue_.add([this, jsonConfigValue] {
        configReceived_ = true;
        const Json::Value startingConfigureSettings = quasar::parseJson(jsonConfigValue);
        auto enabled = quasar::tryGetBool(startingConfigureSettings, "starting_configure_enabled", startingConfigureEnabled_);
        if (startingConfigureEnabled_ != enabled) {
            startingConfigureEnabled_ = enabled;
            YIO_LOG_INFO("starting_configure_enabled: " << (startingConfigureEnabled_ ? "true" : "false"));
        }

        auto connectingTimeout = quasar::tryGetMillis(startingConfigureSettings, "all_services_connecting_timeout_ms", allServicesConnectingTimeout_);
        if (allServicesConnectingTimeout_ != connectingTimeout) {
            allServicesConnectingTimeout_ = connectingTimeout;
            YIO_LOG_INFO("all_services_connecting_timeout_ms: " << allServicesConnectingTimeout_.count());
        }

        auto responseTimeout = quasar::tryGetMillis(startingConfigureSettings, "wait_onboarding_critical_update_directive_timeout_ms", waitResponseDirectiveTimeout_);
        if (waitResponseDirectiveTimeout_ != responseTimeout) {
            waitResponseDirectiveTimeout_ = responseTimeout;
            YIO_LOG_INFO("wait_onboarding_critical_update_directive_timeout_ms: " << waitResponseDirectiveTimeout_.count());
        }
    });
}

void RemoteOnConfigureDoneGreetingObserver::subscribeToSystemConfig() {
    sdk_->subscribeToSystemConfig(CONFIG_NAME);
}

std::weak_ptr<IDemoModeListener> RemoteOnConfigureDoneGreetingObserver::getDemoModeListener() {
    return demoModeListener_;
}
