#include "update_greeting_observer.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

UpdateGreetingObserver::UpdateGreetingObserver(
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    DeviceModeObserver::ConfigurationSuccessState configurationSuccessState,
    std::string soundName,
    const RemoteOnConfigureDoneGreetingObserver::Settings& settings)
    : RemoteOnConfigureDoneGreetingObserver(std::move(sdk), configurationSuccessState, std::move(soundName), settings)
{
}

void UpdateGreetingObserver::onStartSetup(bool isFirstSetup) {
    asyncQueue_.add([this, isFirstSetup] {
        isFirstSetup_ = isFirstSetup;
    });
}

// call under mutex
void UpdateGreetingObserver::requestAction()
{
    YIO_LOG_INFO("requestAction");
    isDirectiveWait_ = true;

    auto request = std::make_shared<YandexIO::VinsRequest>(
        buildStartingOnboardingUpdateSemanticFrame(isFirstSetup_),
        YandexIO::VinsRequest::createSoftwareDirectiveEventSource());

    sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
}

void UpdateGreetingObserver::onSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    RemoteOnConfigureDoneGreetingObserver::onSystemConfig(configName, jsonConfigValue);
    if (configName != CONFIG_NAME) {
        return;
    }

    asyncQueue_.add([this, jsonConfigValue] {
        const Json::Value startingConfigureSettings = quasar::parseJson(jsonConfigValue);
        if (startingConfigureSettings.isMember("wait_onboarding_critical_update_directive_timeout_ms")) {
            auto responseTimeout = quasar::tryGetMillis(startingConfigureSettings, "wait_onboarding_critical_update_directive_timeout_ms", waitResponseDirectiveTimeout_);
            if (waitResponseDirectiveTimeout_ != responseTimeout) {
                waitResponseDirectiveTimeout_ = responseTimeout;
                YIO_LOG_INFO("wait_onboarding_critical_update_directive_timeout_ms: " << waitResponseDirectiveTimeout_.count());
            }
        }
    });
}

Json::Value UpdateGreetingObserver::buildStartingOnboardingUpdateSemanticFrame(bool isFirstSetup) {
    Json::Value semanticFrame;
    Json::Value semanticFramePayload;

    auto& startingCriticalUpdatePayload = semanticFramePayload["typed_semantic_frame"]["onboarding_starting_critical_update"];
    startingCriticalUpdatePayload["is_first_setup"]["bool_value"] = isFirstSetup;

    auto& analytics = semanticFramePayload["analytics"];
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = "get_starting_critical_update_onboarding";

    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";
    semanticFrame["payload"] = semanticFramePayload;

    return semanticFrame;
}
