#include "version_provider.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("version_provider");

using namespace YandexIO;
using namespace quasar;

VersionProvider::VersionProvider(std::shared_ptr<IAuthProvider> authProvider, std::shared_ptr<IDevice> device, std::string version, Settings settings)
    : authProvider_(std::move(authProvider))
    , device_(std::move(device))
    , version_(std::move(version))
    , settings_(std::move(settings))
    , delayTimingsPolicy_(getCrc32(settings_.deviceID) + getNowTimestampMs())
{
    delayTimingsPolicy_.initCheckPeriod(settings_.versionSendBaseSleepFor, settings_.versionSendBaseSleepFor, settings_.versionSendMaxSleepFor);
}

void VersionProvider::provideVersion() {
    callbackQueue_.add([this]() {
        runOncePerBoot(std::bind(&VersionProvider::sendVersion, this), settings_.versionSendGuardFile);
    });
}

std::optional<Json::Value> VersionProvider::readCachedState() {
    if (!fileExists(settings_.cachedVersionFileName)) {
        return std::nullopt;
    }
    try {
        const auto fileContent = getFileContent(settings_.cachedVersionFileName);
        const auto cachedState = parseJson(fileContent);
        if (cachedState.isMember("yandexIOVersion") && cachedState.isMember("environmentVersion")) {
            return cachedState;
        }
        YIO_LOG_ERROR_EVENT("VersionProvider.CachedStateInvalid", "Missing fields in cached state file: " << fileContent);
        return std::nullopt;
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("VersionProvider.CachedStateInvalidJson", "Bad json in cached state file: " << e.what());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("VersionProvider.CachedStateLoadException", "Something went wrong while reading state file: " << e.what());
    }
    return std::nullopt;
}

void VersionProvider::saveState(const Json::Value& version) {
    try {
        AtomicFile cachedStateFile(settings_.cachedVersionFileName);
        cachedStateFile.write(jsonToString(version));
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("VersionProvider.CachedStateSaveException", "Failed to save state in file: " << e.what());
    }
}

bool VersionProvider::sendRequestToBackend() {
    HttpClient client("state-quasar-backend", device_);
    client.setTimeout(std::chrono::milliseconds{60000});
    client.setRetriesCount(0);

    Json::Value request;
    request["state"]["environmentVersion"] = version_;
    request["state"]["yandexIOVersion"] = settings_.softwareVersion;
    auto& currentState = request["state"];

    // read from file. if differ - rewrite file and send
    if (const auto previousState = readCachedState()) {
        if (currentState["yandexIOVersion"] == previousState->get("yandexIOVersion", Json::Value()) &&
            currentState["environmentVersion"] == previousState->get("environmentVersion", Json::Value())) {
            return true;
        }
    }

    auto authInfo = authProvider_->ownerAuthInfo().value();
    const HttpClient::Headers headers = {{"Authorization", "OAuth " + authInfo->authToken}};
    const auto stateUrl = settings_.backendUrl + "/set_device_state?device_id=" + settings_.deviceID + "&platform=" + settings_.platform;
    try {
        const auto response = client.post("set-device-state", stateUrl, jsonToString(request), headers);
        if (response.responseCode == 200) {
            saveState(currentState); // save current state to file. now backend knows it too.
        } else {
            if (response.responseCode == 403 && getStringSafe(response.body, ".message") == "AUTH_TOKEN_INVALID") {
                authProvider_->requestAuthTokenUpdate("State set-device-state");
            }
            YIO_LOG_ERROR_EVENT("VersionProvider.SetDeviceState.Non200Response", "Unable to send device state, backend returned code : " << response.responseCode << ". Body: " << response.body);
        }
        return response.responseCode == 200;
    } catch (const std::runtime_error&) {
        return false;
    }
}

bool VersionProvider::sendVersion() {
    while (!giveUp_.load()) {
        if (sendRequestToBackend()) {
            YIO_LOG_INFO("State sent successfully");
            return true;
        }
        std::this_thread::sleep_for(delayTimingsPolicy_.getDelayBetweenCalls());
        delayTimingsPolicy_.increaseDelayBetweenCalls();
    }
    return false;
}

VersionProvider::~VersionProvider() {
    giveUp_.store(true);
}
