import os
import logging
import shutil
import stat
import subprocess
import sys


logger = logging.getLogger(__name__)


def clean_dir(dir_path):
    try:
        shutil.rmtree(dir_path)
    except OSError:
        pass


class SDKInstaller:
    def __init__(self, java_home, android_sdk, android_sdk_java_home, root):
        self.root = root
        self.java_home = java_home
        self.android_sdk = android_sdk
        self.android_sdk_java_home = android_sdk_java_home

        self.gradle_env = os.environ.copy()
        self.gradle_env['JAVA_HOME'] = os.path.join(root, self.java_home)
        self.gradle_env['JAVA_TOOL_OPTIONS'] = '-Djava.net.preferIPv4Stack=false -Djava.net.preferIPv6Addresses=true'
        self.gradle_env['ANDROID_HOME'] = os.path.join(root, self.android_sdk)
        self.gradle_env['ANDROID_SDK_HOME'] = os.path.join(root, '.android', 'build-cache')

        self.tools_env = os.environ.copy()
        self.tools_env['JAVA_HOME'] = os.path.join(root, self.android_sdk_java_home)
        self.tools_env['JAVA_TOOL_OPTIONS'] = '-Djava.net.preferIPv4Stack=false -Djava.net.preferIPv6Addresses=true'
        self.tools_env['ANDROID_HOME'] = os.path.join(root, self.android_sdk)
        self.tools_env['ANDROID_SDK_HOME'] = os.path.join(root, '.android', 'build-cache')

    def install_android_sdk(self, android_src_dir):
        # Say yes to installer
        subprocess.Popen(
            ['yes | {} --licenses'.format(os.path.join(self.root, self.android_sdk, 'tools', 'bin', 'sdkmanager'))],
            shell=True,
            env=self.tools_env,
            cwd=android_src_dir,
        )

    def get_gradle_env(self):
        return self.gradle_env

    def cleanup(self):
        clean_dir(os.path.join(self.root, self.java_home))
        clean_dir(os.path.join(self.root, self.android_sdk))
        clean_dir(os.path.join(self.root, self.android_sdk_java_home))
        clean_dir(os.path.join(self.root, '.android'))


class GradleWrapper:
    def __init__(self, sdk_installer, android_src_dir):
        self.sdk_installer = sdk_installer
        self.android_src_dir = android_src_dir
        # init empty sign options
        self.gradle_sign_opts = []

    # Enter android_src_dir and fix gradlew script (add +x bit)
    # GradleWrapper expects that gradlew script is located in android_src_dir
    def prepare(self):
        os.chdir(self.android_src_dir)
        # in case of zippatch removed +x bit
        os.chmod(
            os.path.join(self.android_src_dir, 'gradlew'), stat.S_IRWXU | stat.S_IRWXG | stat.S_IROTH | stat.S_IXOTH
        )

    """
    NOTE target apk require to have yandex-signer-oauth propery in build.gradle
        if (hasProperty('yandex-signer-oauth')) {
            apply plugin: 'yandex-signer-android'
            yandexSigner {
                enabled = true
                retryCount = 3
                timeoutSeconds = 300
                signingMethod = "apksigner"
                signingBuildTypes = ['release', 'debug']
            }
        }

    """

    def init_yandex_signer(self):
        yandex_signer = os.environ.get('YANDEX_SIGNER_OAUTH', None)
        if not yandex_signer:
            yandex_signer_file = os.environ.get('YANDEX_SIGNER_FILE', None)
            if yandex_signer_file:
                with open(yandex_signer_file) as f:
                    yandex_signer = f.read()
        if yandex_signer:
            logger.info('yandex-signer auth present, will pass yandex-signer-oauth property')

            self.gradle_sign_opts = ['-Pyandex-signer-oauth={}'.format(yandex_signer)]

    def has_yandex_signer(self):
        return len(self.gradle_sign_opts) > 0

    def clean(self):
        options = [os.path.join(self.android_src_dir, 'gradlew'), 'clean']
        subproc = subprocess.Popen(
            options,
            env=self.sdk_installer.get_gradle_env(),
            cwd=self.android_src_dir,
            stdout=sys.stdout,
            stderr=sys.stderr,
        )
        subproc.communicate()
        return subproc.returncode

    def build(self, target, gradle_opts=None):
        options = [
            os.path.join(self.android_src_dir, 'gradlew'),
            '--stacktrace',
            '--build-cache',
            target,
        ] + self.gradle_sign_opts
        if gradle_opts is not None:
            options += gradle_opts
        subproc = subprocess.Popen(
            options,
            env=self.sdk_installer.get_gradle_env(),
            cwd=self.android_src_dir,
            stdout=sys.stdout,
            stderr=sys.stderr,
        )
        subproc.communicate()
        return subproc.returncode

    def get_result_apk_name(self, prefix):
        # Retrieve result apk
        if self.has_yandex_signer():
            apk_name = '{}-signed.apk'.format(prefix)
        else:
            apk_name = '{}.apk'.format(prefix)
        return apk_name

    def get_result_aar_name(self, prefix):
        # Retrieve result aar
        if self.has_yandex_signer():
            apk_name = '{}-signed.aar'.format(prefix)
        else:
            apk_name = '{}.aar'.format(prefix)
        return apk_name
