import glob
import logging
import argparse
import sys
import os
import shutil

from yandex_io.pylibs.apk_builder.apk_builder import SDKInstaller, GradleWrapper, clean_dir


# TODO: better logging setup maybe?
logging.basicConfig()

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


def main():
    parser = argparse.ArgumentParser(description='Build Jni APK.')
    parser.add_argument(
        '--android-src-dir', metavar='path', type=str, required=True, help='path to source dir of android app'
    )
    parser.add_argument('--java-home', metavar='path', type=str, required=True, help='path to jdk')
    parser.add_argument('--android-sdk', metavar='path', type=str, required=True, help='path to android sdk')
    parser.add_argument(
        '--android-sdk-java-home',
        metavar='path',
        type=str,
        required=False,
        help='path to android sdk java home',
    )
    parser.add_argument(
        '--external-build-src',
        metavar='path',
        type=str,
        help='path to external_build data (jni libs, .jar, etc)',
        default=None,
    )
    parser.add_argument(
        '--external-build-dst', metavar='path', type=str, help='destination of external_build data', default=None
    )

    parser.add_argument(
        '-p', '--product', type=str, required=False, help='Gradle product to use for build.', default='all'
    )
    parser.add_argument(
        '-t',
        '--task',
        action='append',
        help='Gradle tasks to build execute. Specify multiple times if needed, will be run in that order.',
    )
    parser.add_argument(
        '--property',
        action='extend',
        nargs="+",
        type=str,
        help='Add gradle property. Specify multiple times if needed.'
    )
    parser.add_argument(
        '-o',
        '--output',
        action='append',
        default=[],
        help='Output files to copy to the build root. ' 'Specify multiple times if needed, all will be copied.' 'Unix-style path globs like * are expanded',
    )
    parser.add_argument(
        '--output-renamed',
        action='append',
        default=[],
        help='Same as --output, but with rename -- give in format <path/to/source:name_in_buildroot> ' 'Unix-style path globs like * are expanded, but should result in single file'
    )
    parser.add_argument(
        '--enable-yandex-signer',
        action='store_true',
        default=False,
        help='Enable yandex_signer for this build. '
        'Relies on build.gradle handling of yandex-signer-oauth property '
        'AND either YANDEX_SIGNER_OAUTH or YANDEX_SIGNER_FILE env vars present.',
    )
    parser.add_argument(
        '--enable-appmetrica-plugin-online',
        action='store_true',
        default=False,
        help='Enable load mapping.txt classes to appmetrica.',
    )
    parser.add_argument(
        '--disable-clean', action='store_true', default=False, help='Disable intrinsic gradle clean action'
    )
    parser.add_argument('--revision', type=int, required=False, help='revision for centaur build version.')
    parser.add_argument('--branch', type=str, required=False, help='branch for centaur build version.')

    args = parser.parse_args()

    root = os.getcwd()

    android_src_dir = os.path.join(root, args.android_src_dir)

    # Move external build (jni libs and data) to apk sources
    # FIXME: this part is confusing, needs rework
    external_build_src = args.external_build_src
    if external_build_src is not None:
        external_build_dst = args.external_build_dst

        if external_build_dst is not None:
            src = os.path.join(root, external_build_src)
            dst = os.path.join(root, external_build_dst)
            logger.info('Paths %s %s' % (src, dst))
            for item in os.listdir(src):
                shutil.move(os.path.join(src, item), dst)
        else:
            shutil.move(os.path.join(root, external_build_src), android_src_dir)

    # Init SDKInstaller (set up gradle environment)
    sdk_installer = SDKInstaller(args.java_home, args.android_sdk, args.android_sdk_java_home, root)
    sdk_installer.install_android_sdk(android_src_dir)

    gradle_wrapper = GradleWrapper(sdk_installer, android_src_dir)

    if args.enable_yandex_signer:
        logger.info('Enabled yandex signer')

        gradle_wrapper.init_yandex_signer()

    # Enter android sources dir and fix gradlew script (cmhod +x)
    gradle_wrapper.prepare()

    if not args.disable_clean:
        logger.info('Clean is not disabled, performing it...')
        res = gradle_wrapper.clean()
        if res != 0:
            logger.error('Failed to clean')

            sys.exit(res)

    # FIXME: move to the arguments
    gradle_opts = ['-Pdisable-external-builder=true', '-Pproduct=' + args.product]
    if args.revision:
        gradle_opts.append('-Prevision=' + str(args.revision))

    if args.branch:
        gradle_opts.append('-Pbranch=' + args.branch)

    if args.enable_appmetrica_plugin_online:
        gradle_opts.append('-Penable-appmetrica-plugin-online')

    if args.property:
        gradle_opts.extend((f'-P{x}' for x in args.property))

    for task in args.task:
        logger.info('Building task %s with opts %s' % (task, gradle_opts))
        res = gradle_wrapper.build(task, gradle_opts)

        if res != 0:
            logger.error('Task %s failed with exit code %d, terminating' % (task, res))
            sys.exit(res)

    for output in args.output:
        for item in glob.glob(os.path.join(android_src_dir, output)):
            shutil.copy(item, root)

    for output in args.output_renamed:
        src, target = output.split(':')

        items = glob.glob(os.path.join(android_src_dir, src))

        if len(items) > 1:
            logger.error('more than one file matched glob %s: %s' % (src, items))
            sys.exit(1)

        if items:
            shutil.copy(items[0], os.path.join(root, target))

    # clean package. It should contain only result artifact
    clean_dir(android_src_dir)
    clean_dir(os.path.join(root, 'speechkit'))
    clean_dir(os.path.join(root, 'quasar'))
    clean_dir(os.path.join(root, 'smart_devices'))
    clean_dir(os.path.join(root, 'yandex_io'))
    sdk_installer.cleanup()


if '__main__' == __name__:
    sys.exit(main())
