import requests
import subprocess
import logging
import time

logging.basicConfig()
logger = logging.getLogger(__name__)


def _fail(message):
    logger.error(message)
    raise RuntimeError(message)


class KolhozDevice:
    def __init__(self, device_id, kolhoz_device_id, kolhoz_token):
        self._device_id = device_id
        self._kolhoz_id = kolhoz_device_id
        self._device_addr = None
        self._headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth {}'.format(kolhoz_token)}

    @property
    def kolhoz_id(self):
        return self._kolhoz_id

    @property
    def device_id(self):
        return self._device_id

    @staticmethod
    def get_device(kolhoz_token, manufacturer_name=None, group=None, device_id=None, booking_time_min=120):
        filters = {
            'status': 'READY',
            'osType': 'STATION',
            'deviceType': 'STATION',
            'connectivityState': True,
        }
        if manufacturer_name is not None and manufacturer_name != "":
            filters["manufacturerText"] = manufacturer_name
        if device_id is not None and device_id != "":
            filters["udids"] = device_id
        if group is not None and group != "":
            filters["group"] = group

        headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth {}'.format(kolhoz_token)}
        cur_ts_ms = int(time.time() * 1000)
        options = {"fromTs": cur_ts_ms, "toTs": cur_ts_ms + booking_time_min * 60 * 1000}
        body={'filter': filters, 'options': options},
        logger.info("Try to get device. Body: {}".format(body))
        resp = requests.post(
            "https://kolhoz.yandex-team.ru/api/public/v1/devices/occupy",
            headers=headers,
            json={'filter': filters, 'options': options},
        )
        if resp.status_code != 200:
            _fail(
                "Failed to get device with {}. Error code: {}. Error message: {}".format(
                    filters, resp.status_code, resp.text
                )
            )
        json_resp = resp.json()
        if not json_resp['results']:
            _fail("Get empty list from occupy request")
        device_id = json_resp['results'][0]['device']['udid']
        kolhoz_id = json_resp['results'][0]['device']['id']
        logger.info("Got device: {}, internal id: {}".format(device_id, kolhoz_id))
        device = KolhozDevice(device_id, kolhoz_id, kolhoz_token)
        return device

    def connect(self):
        host, port = self._enable_remote_connect()
        self._device_addr = host + ":" + str(port)
        time.sleep(1)
        self._connect()
        time.sleep(5)

    def reconnect(self):
        self._connect()

    def remount(self):
        self._execute_shell_command(["adb", "-s", self._device_addr, "remount"])

    def _enable_remote_connect(self):
        resp = requests.post(
            'https://kolhoz.yandex-team.ru/api/public/v1/devices/' + self._kolhoz_id + '/connect', headers=self._headers
        )
        if resp.status_code != 200:
            _fail(
                "Failed to enable remote connect to device: {}. Error code: {}. Error message: {}".format(
                    self._device_id, resp.status_code, resp.text
                )
            )
        json_resp = resp.json()
        logger.info(json_resp)
        if not json_resp["enabled"]:
            _fail("Failed to enable remote connect to device: {}. Response: {}".format(self._kolhoz_id, json_resp))
        return json_resp["hostname"], json_resp["port"]

    def _connect(self):
        self._execute_shell_command(["adb", "connect", self._device_addr])

    @staticmethod
    def _execute_shell_command(cmd):
        logger.info('\nExecuting: {}'.format(' '.join(cmd)))
        stdout, stderr = subprocess.Popen(
            cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, universal_newlines=True
        ).communicate()
        if stdout:
            logger.info("stdout:\n{}".format(stdout.strip()))
        if stderr:
            logger.info("stderr:\n{}".format(stderr.strip()))

    def shell(self, command):
        if self._device_addr is None:
            logger.error("Connect to device before using shell commands.")
            return
        self._execute_shell_command(["adb", "-s", self._device_addr, "shell"] + command.split())

    def _disconnect(self):
        if self._device_addr is None:
            return
        self._execute_shell_command(["adb", "disconnect", self._device_addr])

    def _release_device(self):
        resp = requests.post(
            'https://kolhoz.yandex-team.ru/api/public/v1/devices/' + self._kolhoz_id + '/release', headers=self._headers
        )
        if resp.status_code != 200:
            _fail(
                "Failed to release device with divice_id: {}. Error code: {}. Error message: {}".format(
                    self._device_id, resp.status_code, resp.text
                )
            )

    def disconnect(self):
        self._disconnect()
        self._release_device()

    def forward(self, port1, port2):
        self._execute_shell_command(
            ["adb", "-s", self._device_addr, "forward", "tcp:{}".format(port1), "tcp:{}".format(port2)]
        )

    def push(self, src, dst):
        self._execute_shell_command(["adb", "-s", self._device_addr, "push", src, dst])

    def pull(self, src, dst):
        self._execute_shell_command(["adb", "-s", self._device_addr, "pull", src, dst])
