import requests
import logging
import json


class QuasmodromClient:
    URL = "https://quasmodrom.yandex-team.ru"

    def __init__(self, token):
        self._token = token

    def _validate_resp(self, resp):
        try:
            resp.raise_for_status()
        except:
            raise RuntimeError("Backend request failed: {} {}".format(resp.status_code, resp.content))

    def update_group(self, model, firmware_id, group_id, critical=False):
        headers = {"Authorization": "OAuth {}".format(self._token), "Content-type": "application/x-www-form-urlencoded"}
        data = "firmware={}&group={}&critical={}".format(firmware_id, group_id, "true" if critical else "false")
        resp = requests.post("/".join([self.URL, model, "update"]), headers=headers, data=data)
        self._validate_resp(resp)
        url = resp.json()["url"]
        logging.info("Created firmware update: " + url)
        return url

    def get_last_firmware_update(self, model, group_name):
        resp = requests.get("/".join([self.URL, model, "group", group_name, "last_firmware"]))
        self._validate_resp(resp)
        resp_json = resp.json()
        return resp_json["firmware_id"], resp_json["version"]

    def set_device_settings(self, model, device_id, firmware_id=None, config=None, group=None):
        if config is not None:
            config = json.dumps(config)
        headers = {"Authorization": "OAuth {}".format(self._token), "Content-type": "application/x-www-form-urlencoded"}
        params = [
            param
            for param in (
                ("device_id", device_id),
                ("forced_firmware", firmware_id),
                ("config", config),
                ("group", group),
            )
            if param[1] is not None
        ]
        data = "&".join(map(lambda param: "{}={}".format(param[0], param[1]), params))
        logging.info(data)
        resp = requests.patch("/".join([self.URL, model, "device"]), headers=headers, data=data)
        self._validate_resp(resp)
        url = resp.json()["url"]
        logging.info("Created device config update: " + url)
        return url
