#include "sample_hal.h"
#include "sample_volume_manager.h"

#include <yandex_io/sample_app/linux/audio_service/audio_source_service.h>

#include <yandex_io/capabilities/volume/volume_capability.h>

#include <yandex_io/libs/self_destroyer/self_destroyer_utils.h>
#include <yandex_io/metrica/monitor/monitor_service.h>
#include <yandex_io/modules/audio_input/audio_device/tcp/tcp_audio_device.h>
#include <yandex_io/modules/bluetooth/bluetooth_module.h>
#include <yandex_io/modules/configuration_mode_sound/configuration_mode_handler.h>
#include <yandex_io/modules/equalizer_controller/dispatcher/gstreamer/gstreamer_equalizer_dispatcher.h>
#include <yandex_io/modules/equalizer_controller/equalizer_controller.h>
#include <yandex_io/modules/geolocation/util/util.h>
#include <yandex_io/modules/keyboard_manager/keyboard_manager.h>
#include <yandex_io/modules/starting_greeting/default_on_configure_done_greeting_observer.h>
#include <yandex_io/modules/starting_greeting/starting_greeting_data.h>
#include <yandex_io/modules/testpoint/bluetooth_emulator.h>
#include <yandex_io/modules/testpoint/testpoint_service.h>
#include <yandex_io/modules/xiva_alice_request/xiva_push_alice_requester.h>
#include <yandex_io/modules/personalization/bio_capability_switcher.h>
#include <yandex_io/modules/spotter_configurer/spotter_configurer.h>

#include <yandex_io/scaffolding/scaffolding.h>
#include <yandex_io/scaffolding/base/endpoint.h>
#include <yandex_io/scaffolding/base/entry.h>
#include <yandex_io/scaffolding/mediad/mediad.h>
#include <yandex_io/sdk/device_mode_observer.h>
#include <yandex_io/sdk/yandex_iosdk.h>
#include <yandex_io/sdk/hub/io_hub_service.h>
#include <yandex_io/sdk/private/endpoint_storage/endpoint.h>
#include <yandex_io/services/alarmd/alarm_service.h>
#include <yandex_io/services/aliced/alice_service.h>
#include <yandex_io/services/authd/auth_service.h>
#include <yandex_io/services/ble_initd/ble_service.h>
#include <yandex_io/services/braind/brain_service.h>
#include <yandex_io/services/brickd/brick_service.h>
#include <yandex_io/services/bug_report/bug_report_service.h>
#include <yandex_io/services/do_not_disturb/do_not_disturb_service.h>
#include <yandex_io/services/firstrund/first_run_service.h>
#include <yandex_io/services/glagold/glagol.h>
#include <yandex_io/services/glagold/mdnsd_nsd_messager.h>
#include <yandex_io/services/iot/iot_service.h>
#include <yandex_io/services/mediad/audio_clock_manager/audio_clock_manager.h>
#include <yandex_io/services/mediad/media_service.h>
#include <yandex_io/services/networkd/network_service.h>
#include <yandex_io/services/notificationd/notification_service.h>
#include <yandex_io/services/ntpd/ntp_service.h>
#include <yandex_io/services/pushd/push_service.h>
#include <yandex_io/services/setupd/setup_service.h>
#include <yandex_io/services/sound_initd/sound_init_service.h>
#include <yandex_io/services/stereo_paird/stereo_pair_service.h>
#include <yandex_io/services/syncd/sync_service.h>
#ifndef __ANDROID__
    #include <yandex_io/modules/audio_input/audio_device/alsa/alsa_audio_device.h>
#endif

#include <yandex_io/libs/audio_player/factory/player_factory_selector.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/asio/asio_ipc_factory.h>
#include <yandex_io/libs/ipc/datacratic/datacratic_ipc_factory.h>
#include <yandex_io/libs/ipc/mixed/mixed_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/telemetry/file/file_telemetry.h>
#include <yandex_io/libs/telemetry/null/null_metrica.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/speechkit_audio_player/devnull_audio_player.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/clock_tower/connector/clock_tower_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/glagol/connector/glagol_cluster_provider.h>
#include <yandex_io/interfaces/multiroom/connector/multiroom_provider.h>
#include <yandex_io/interfaces/spectrogram_animation/connector/spectrogram_animation_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/interfaces/updates/connector/updates_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/interfaces/volume_manager/connector/volume_manager_provider.h>

#include <speechkit/PlatformTypes.h>
#include <library/cpp/testing/common/env.h>

#include <util/generic/size_literals.h>
#include <util/system/user.h>

#include <boost/program_options.hpp>

#include <memory>
#include <optional>

using namespace quasar;
using namespace quasar::proto;

namespace {

    std::shared_ptr<ipc::IIpcFactory> createIpcFactory(std::shared_ptr<YandexIO::IDevice> device, const std::string& engine) {
        if (engine == "mixed") {
            auto context = ipc::MixedIpcFactory::createDefaultContext(device->sharedConfiguration());
            return std::make_shared<ipc::MixedIpcFactory>(context, "maind");
        } else if (engine == "datacratic") {
            return std::make_shared<ipc::DatacraticIpcFactory>(device->sharedConfiguration());
        } else if (engine == "asio") {
            return std::make_shared<ipc::AsioIpcFactory>(device->sharedConfiguration());
        } else {
            throw std::runtime_error("unknown IPC engine");
        }
    }

    std::unique_ptr<YandexIO::AudioDevice> openAudioDevice(std::optional<std::string> deviceParam, const Json::Value& config) {
        std::string device;
        if (deviceParam) {
            device = *deviceParam;
        } else if (config.isMember("audioDevice")) {
            device = tryGetString(config["audioDevice"], "device", "alsa");
        } else {
            device = "alsa";
        }

        if (device == "alsa") {
#ifndef __ANDROID__
            return std::make_unique<YandexIO::AlsaAudioDevice>(config);
#else
            throw std::logic_error("ALSA is not supported on Android");
#endif
        } else if (device == "tcp") {
            return std::make_unique<YandexIO::TCPAudioDevice>(config);
        } else {
            throw std::logic_error("Unknown audio device type");
        }
    }

    std::string get_working_path() {
        char temp[MAXPATHLEN];
        return getcwd(temp, sizeof(temp)) ? std::string(temp) : std::string("");
    }

    std::unique_ptr<YandexIO::ITelemetry> makeTelemetry(const std::optional<std::string>& telemetryFile) {
        if (telemetryFile) {
            return std::make_unique<YandexIO::FileTelemetry>(*telemetryFile);
        } else {
            return std::make_unique<NullMetrica>();
        }
    }

    // make device id, that will apend USER, so it will not intersect btw developers
    YandexIO::DeviceID makeUserDeviceId(YandexIO::Configuration* configuration) {
        const auto config = configuration->getServiceConfig("common");
        const auto deviceIdFileName = getString(config, "deviceIdFileName");
        auto deviceId = boost::trim_copy(getFileContent(deviceIdFileName));
        if (tryGetBool(config, "appendUserToDeviceId", false)) {
            const auto user = GetUsername();
            deviceId = deviceId + "-" + user;
        }
        return deviceId;
    }

    std::shared_ptr<YandexIO::IDevice> makeDevice(const std::string& configFile,
                                                  const YandexIO::ConfigPatterns& extra,
                                                  const std::optional<std::string>& telemetryFile,
                                                  const std::optional<std::string> optDeviceId) {
        auto configuration = YandexIO::makeConfiguration(configFile, extra);
        auto deviceId = optDeviceId.has_value() ? *optDeviceId : makeUserDeviceId(configuration.get());
        auto telemetry = makeTelemetry(telemetryFile);
        auto hal = std::make_unique<SampleHAL>();
        return std::make_shared<YandexIO::Device>(
            std::move(deviceId),
            std::move(configuration),
            std::move(telemetry),
            std::move(hal));
    }

    SpeechKit::AudioPlayer::SharedPtr createAudioPlayer(YandexIO::IDevice& device) {
        const auto config = device.configuration()->getServiceConfig("aliced");
        const auto playerString = tryGetString(config, "audioPlayer", "speechkit");
        SpeechKit::AudioPlayer::SharedPtr player;
        if (playerString == "speechkit") {
            const bool closeOpenALWhenIdle = tryGetBool(config, "closeOpenALWhenIdle", false);
            player = ::SpeechKit::PlatformTypes::createDefaultAudioPlayer(closeOpenALWhenIdle);
        } else if (playerString == "devnull") {
            player = DevnullAudioPlayer::create();
        } else {
            throw std::logic_error("Unknown aliced player type " + playerString);
        }
        return player;
    }

    // Fake Wifid that will send ACK to requests (make firstrund and sound_initd happy)
    class FakeWifiService: public QuasarService {
        std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        std::string serviceName_;
        std::shared_ptr<ipc::IServer> server_;

    public:
        explicit FakeWifiService(std::shared_ptr<ipc::IIpcFactory> ipcFactory, std::string name)
            : ipcFactory_(std::move(ipcFactory))
            , serviceName_(std::move(name))
        {
        }

        std::string getServiceName() const override {
            return serviceName_;
        }

        void start() override {
            server_ = ipcFactory_->createIpcServer(serviceName_);
            server_->setMessageHandler([](const auto& msg, auto& connection) {
                QuasarMessage response;
                response.set_request_id(msg->request_id());
                if (msg->has_wifi_list_request()) {
                    response.mutable_wifi_list();
                } else if (msg->has_wifi_connect()) {
                    auto wifi = response.mutable_wifi_connect_response();
                    wifi->set_status(proto::WifiConnectResponse::OK);
                }
                connection.send(std::move(response));
            });
            server_->listenService();
        }
    };

    class FakeFirstrund: public QuasarService {
        std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        std::shared_ptr<ipc::IServer> server_;

    public:
        explicit FakeFirstrund(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
            : ipcFactory_(std::move(ipcFactory))
        {
        }

        std::string getServiceName() const override {
            return "firstrund";
        }

        void start() override {
            const auto serviceName = getServiceName();
            server_ = ipcFactory_->createIpcServer(serviceName);
            server_->setClientConnectedHandler([](auto& connection) {
                QuasarMessage msg;
                msg.set_configuration_state(proto::ConfigurationState::CONFIGURED);
                connection.send(std::move(msg));
            });
            server_->listenService();
        }
    };

    class GlagolServiceWrapper: public QuasarService {
    public:
        using NeighborsList = std::vector<std::tuple<std::string, int>>;

        explicit GlagolServiceWrapper(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
            std::shared_ptr<IMultiroomProvider> multiroomProvider,
            std::shared_ptr<IStereoPairProvider> stereoPairProvider,
            INsdMessagerFactory& nsdMessagerFactory,
            const NeighborsList& neighbors,
            std::shared_ptr<YandexIO::YandexIOSDK> sdk)
            : device_(std::move(device))
            , ipcFactory_(std::move(ipcFactory))
            , authProvider_(std::move(authProvider))
            , deviceStateProvider_(std::move(deviceStateProvider))
            , multiroomProvider_(std::move(multiroomProvider))
            , stereoPairProvider_(std::move(stereoPairProvider))
            , nsdMessagerFactory_(nsdMessagerFactory)
            , neighbors_(neighbors)
            , sdk_(std::move(sdk))
        {
        }

        std::string getServiceName() const override {
            return "glagol";
        }

        class DummyResponderStatus: public ResponderStatus {
            void pollCompleted(std::string /*lastState*/) override{};
            std::string getResponderStatusJson() override {
                return "{\"development\":true}";
            };
            bool isNetworkAvailable() const override {
                return true;
            };
        };

        class FixedMDNSHolder: public MDNSHolder {
            std::mutex mutex_;
            std::condition_variable cond_;
            const NeighborsList neighbors_;
            std::shared_ptr<ResponderStatus> responderStatus_;
            ResolveHandler* resolveHandler_;
            std::thread thread_;
            std::atomic_bool stopped_{false};
            using ResolveItem = glagol::ResolveItem;

            static std::tuple<ResolveItem, ResolveItem::Info> makeResolveInfo(const std::string& id, int port) {
                ResolveItem item{
                    .name = "YandexIOReceiver-" + id,
                    .type = "_yandexio._tcp",
                    .domain = "local",
                };
                ResolveItem::Info info{
                    .hostname = id,
                    .port = port,
                    .address = "localhost",
                };

                info.txt = {
                    {"deviceId", id},
                    {"platform", "development"},
                    {"cluster", "yes"}};

                return std::make_tuple(item, info);
            };

            void anounce() {
                for (const auto& [id, port] : neighbors_) {
                    auto [item, info] = makeResolveInfo(id, port);
                    resolveHandler_->newResolve(item, info);
                }
            }

        public:
            FixedMDNSHolder(const NeighborsList neighbors,
                            ResolveHandler* resolveHandler)
                : neighbors_(neighbors)
                , responderStatus_(std::make_shared<DummyResponderStatus>())
                , resolveHandler_(resolveHandler)
            {
                thread_ = std::thread(&FixedMDNSHolder::threadLoop, this);
            };

            ~FixedMDNSHolder() {
                {
                    std::scoped_lock<std::mutex> lock(mutex_);
                    stopped_ = true;
                    cond_.notify_one();
                }
                thread_.join();
            }

            void threadLoop() {
                std::unique_lock<std::mutex> lock(mutex_);
                while (!stopped_) {
                    anounce();
                    cond_.wait_for(
                        lock,
                        std::chrono::seconds(60),
                        [this]() {
                            return stopped_.load();
                        });
                }
            }

            std::shared_ptr<ResponderStatus> getResponderStatusProvider() override {
                return responderStatus_;
            }
            void setNetworkAvailability(bool /*isAvailable*/) override {
            }
            void reconfigure() override {
            }
        };

        MDNSHolderFactory makeMdnsFactory() {
            if (neighbors_.empty()) {
                return createDefaultMndsFactory(nsdMessagerFactory_);
            }

            return [this](std::shared_ptr<YandexIO::IDevice> /*device*/, const MdnsSettings& /*settings*/, ResolveHandler* resolveHandler, MDNSTransportOpt /*options*/) -> MDNSHolderPtr {
                return std::make_shared<FixedMDNSHolder>(neighbors_, resolveHandler);
            };
        }

        void start() override {
            const auto config = device_->configuration()->getServiceConfig("glagold");
            glagol_ = std::make_shared<Glagol>(
                device_,
                ipcFactory_,
                authProvider_,
                deviceStateProvider_,
                multiroomProvider_,
                stereoPairProvider_,
                makeMdnsFactory(),
                config,
                Glagol::Settings(),
                sdk_);
            sdk_->getEndpointStorage()->getLocalEndpoint()->addListener(glagol_);
        }

    private:
        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        std::shared_ptr<IAuthProvider> authProvider_;
        std::shared_ptr<IDeviceStateProvider> deviceStateProvider_;
        std::shared_ptr<IMultiroomProvider> multiroomProvider_;
        std::shared_ptr<IStereoPairProvider> stereoPairProvider_;
        INsdMessagerFactory& nsdMessagerFactory_;
        NeighborsList neighbors_;
        std::shared_ptr<quasar::Glagol> glagol_;
        std::shared_ptr<YandexIO::YandexIOSDK> sdk_;
    };

    std::unique_ptr<YandexIO::EqualizerDispatcherFactory> createEqualizerDispatcherFactory(std::weak_ptr<YandexIO::SDKInterface> sdk) {
        auto factory = std::make_unique<YandexIO::EqualizerDispatcherFactory>();
        factory->addDispatcherType("gstreamer", [sdk{std::move(sdk)}]() {
            return std::make_unique<YandexIO::GStreamerEqualizerDispatcher>(sdk);
        });
        return factory;
    }

} // anonymous namespace

namespace std {
    std::istream& operator>>(std::istream& inputStream, std::tuple<std::string, int>& dst) {
        std::string tmp;
        inputStream >> tmp;
        auto pos = tmp.find(':');
        if (pos == std::string::npos) {
            throw std::runtime_error("Invalid format ':' is expected");
        }
        std::get<1>(dst) = std::atoi(tmp.data() + pos + 1);
        tmp.resize(pos);
        std::get<0>(dst) = tmp;
        return inputStream;
    }

    std::istream& operator>>(std::istream& inputStream, std::optional<std::string>& dst) {
        std::string str;
        inputStream >> str;
        dst = std::move(str);
        return inputStream;
    }
} // namespace std

int main(int argc, char** argv)
{
    quasar::TerminateWaiter waiter;

    /**
     * Init Curl before any thread run. It is necessary because curl_global_init is NOT THREAD SAFE
     */
    quasar::HttpClient::globalInit();

    auto config = std::make_shared<LauncherConfig>();

    const auto workingDirectory = get_working_path() + "/workdir";

    YandexIO::ConfigPatterns extra;
    extra["SOURCE_ROOT"] = ArcadiaSourceRoot();
    extra["DATA"] = workingDirectory;
    extra["TMP"] = workingDirectory;
    extra["LOGS"] = workingDirectory + "/logs";

    const std::string defaultConfigFile = ArcadiaSourceRoot() + "/yandex_io/sample_app/linux/config/quasar.cfg";
    std::optional<std::string> deviceId;

    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "produce help message")("config", boost::program_options::value<std::string>()->default_value(defaultConfigFile), "path to quasar.cfg")("config_pattern", boost::program_options::value<std::vector<std::string>>(), "override patterns in config files, separated by :, can be used multiple times, e.g. --config_pattern PATH:VALUE --config_pattern DATA:DATA_VALUE")("authcode", boost::program_options::value<std::string>(), "xcode used to get oauth token (see tools/get_oauth_code for getting it)")("telemetry_file", boost::program_options::value<std::string>(), "path to a file that will have all of the telemetry dumped into, instead of sending it to appmetrica")("logs-to-stdout", boost::program_options::value<std::string>()->implicit_value("debug"), "configure logs to stdout. Arg is logLevel (trace, debug, info, error)")("logs-to-file", boost::program_options::value<std::string>()->implicit_value("debug"), "configure logs to `sample_app.log` file. Arg is logLevel (trace, debug, info, error)")("ipc-engine", boost::program_options::value<std::string>()->default_value("mixed"), "IPC implementation to use (mixed|datacratic|asio)")("fileno-limit", boost::program_options::value<uint32_t>()->default_value(3072), "fd limit (sample_app required fd limit > 2048)")("use-glagol", boost::program_options::bool_switch()->default_value(false), "force use of real glagol service (with avahi used)")("neighbor", boost::program_options::value<std::vector<std::tuple<std::string, int>>>(), "Disable avahi. Specifiy network neighbor to be discovered by glagol --neighbor ID1:PORT1 [ID2:PORT2 ...]")("disable-key-input", boost::program_options::bool_switch()->default_value(false), "Disable keyboard input")("skip-register", boost::program_options::bool_switch()->default_value(false), "Use Fake Firstrund which will send always CONFIGURED state")("audio", boost::program_options::value<std::string>(), "Override audio device from config")("device-id", boost::program_options::value(&deviceId), "Override automatically generated device ID");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);
    if (opts.count("help"))
    {
        std::cout << desc << "\n";
        return 0;
    }

    try {
        boost::program_options::notify(opts);
    } catch (const boost::program_options::required_option& e) {
        std::cout << "Error: " << e.what() << "\n";
        std::cout << desc << "\n";
        return -1;
    }

    if (opts.count("config_pattern")) {
        std::vector<std::string> kvs = opts["config_pattern"].as<std::vector<std::string>>();
        for (const auto& kv : kvs) {
            extra[kv.substr(0, kv.find(':'))] = kv.substr(kv.find(':') + 1);
        }
    }

    std::optional<std::string> telemetryFile = std::nullopt;
    if (opts.count("telemetry_file")) {
        telemetryFile = opts["telemetry_file"].as<std::string>();
    }

    std::optional<std::string> audioDeviceParam = std::nullopt;
    if (opts.count("audio")) {
        audioDeviceParam = opts["audio"].as<std::string>();
    }

    std::string logLevel;
    if (opts.count("logs-to-stdout")) {
        logLevel = opts["logs-to-stdout"].as<std::string>();
        quasar::Logging::initLoggingToStdout(logLevel);
    } else if (opts.count("logs-to-file")) {
        logLevel = opts["logs-to-file"].as<std::string>();
        quasar::Logging::initLoggingToFile("sample_app.log", logLevel, 1_GB, 2);
    }

    config->mutable_paths()->set_workdir_path(TString(extra["DATA"]));
    config->mutable_paths()->set_config_path(TString(opts["config"].as<std::string>()));
    if (opts.count("authcode")) {
        config->set_auth_code(TString(opts["authcode"].as<std::string>()));
    }
    config->mutable_limits()->set_fileno_limit(opts["fileno-limit"].as<uint32_t>());
    config->set_ca_certificates_path("/etc/ssl/certs/ca-certificates.crt");
    YIO_LOG_INFO("Preparing config");
    prepareEnvironment(*config);
    YIO_LOG_INFO("Make device");

    auto device = makeDevice(config->paths().config_path(), extra, telemetryFile, deviceId);
    YIO_LOG_INFO("Create maind instance");
    QuasarMinidump::getInstance().init("maind", device->configuration()->getServiceConfig("maind"), device->telemetry(), device->softwareVersion());

    YIO_LOG_INFO("Create IPC factory");
    auto ipcFactory = createIpcFactory(device, opts["ipc-engine"].as<std::string>());
    if (logLevel.empty()) {
        // TODO: move logging setup to the top of main() (before any background threads)
        const auto maindConfig = device->configuration()->getServiceConfig("yiod");
        quasar::Logging::initLogging(maindConfig);
    }

    YIO_LOG_INFO("init sdk");
    const auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, "sample_app", device->deviceId());

    // Module that provides location and timezone to services via SDK
    const auto [geoModule, geoSDKSetter] = YandexIO::GeolocationUtil::installModule(sdk, device, workingDirectory);

    quasar::YandexIOEndpoint endpoint(device, "yiod");

    const bool isMultiroomSupported = device->configuration()->isFeatureSupported("multiroom");
    const bool isStereoPairSupported = device->configuration()->isFeatureSupported("stereo_pair");

    auto authProvider = std::make_shared<AuthProvider>(ipcFactory);
    auto clockTowerProvider = (isMultiroomSupported ? std::make_shared<ClockTowerProvider>(ipcFactory) : nullptr);
    auto deviceStateProvider = std::make_shared<DeviceStateProvider>(ipcFactory);
    auto glagolClusterProvider = std::make_shared<GlagolClusterProvider>(ipcFactory);
    auto multiroomProvider = (isMultiroomSupported ? std::make_shared<MultiroomProvider>(device, ipcFactory) : nullptr);
    auto spectrogramAnimationProvider = std::make_shared<SpectrogramAnimationProvider>(ipcFactory);
    auto stereoPairProvider = (isStereoPairSupported ? std::make_shared<StereoPairProvider>(device, ipcFactory) : nullptr);
    auto updatesProvider = std::make_shared<UpdatesProvider>(ipcFactory);
    auto userConfigProvider = std::make_shared<UserConfigProvider>(ipcFactory);
    auto volumeManagerProvider = std::make_shared<VolumeManagerProvider>(ipcFactory);
    auto delayTimingsPolicy = std::make_shared<BackoffRetriesWithRandomPolicy>(getCrc32(device->deviceId()) + getNowTimestampMs());

    const auto commonConfig = device->configuration()->getCommonConfig();
    const auto currentVolumeFile = getString(commonConfig, "currentVolumeFile");
    const auto currentMuteStateFile = getString(commonConfig, "currentMuteStateFile");
    auto volumeManager = std::make_shared<SampleVolumeManager>(device, ipcFactory, sdk, currentVolumeFile, currentMuteStateFile);
    MdnsdMessagerFactory nsdMessagerFactory{ipcFactory};

    endpoint.addService<YandexIO::IOHubService>(ipcFactory);
    endpoint.addService<BrainService>(device, ipcFactory);
    endpoint.addService<DoNotDisturbService>(device, ipcFactory);
    endpoint.addService<PushService>(device, ipcFactory, authProvider, userConfigProvider);
    endpoint.addService<SyncService>(device, ipcFactory, authProvider, deviceStateProvider, delayTimingsPolicy);
    endpoint.addService<AlarmService>(device, ipcFactory, authProvider, stereoPairProvider);
    endpoint.addService<BrickService>(device, ipcFactory);
    endpoint.addService<BleService>(device, ipcFactory, deviceStateProvider);

    endpoint.addService<AliceService>(
        device,
        NAlice::TEndpoint::SpeakerEndpointType,
        sdk,
        ipcFactory,
        authProvider,
        clockTowerProvider,
        deviceStateProvider,
        glagolClusterProvider,
        multiroomProvider,
        stereoPairProvider,
        userConfigProvider,
        createAudioPlayer(*device),
        SelfDestroyerUtils::createStub(device));

    auto bluetoothEmulator = std::make_shared<YandexIO::BluetoothEmulator>();
    YandexIO::BluetoothModule::Settings settings;
    YandexIO::BluetoothModule btModule(bluetoothEmulator, stereoPairProvider, sdk, settings);

    if (device->configuration()->hasServiceConfig("testpoint")) {
        endpoint.addService<TestpointService>(device, sdk, ipcFactory, volumeManager, bluetoothEmulator);
    }

    endpoint.addService<SoundInitService>(device, ipcFactory, deviceStateProvider, sdk);
    endpoint.addService<AuthService>(device, ipcFactory);
    if (opts["skip-register"].as<bool>()) {
        endpoint.addService<FakeFirstrund>(ipcFactory);
    } else {
        endpoint.addService<FirstRunService>(device, ipcFactory, authProvider, deviceStateProvider, updatesProvider, userConfigProvider, sdk);
    }
    endpoint.addService<IotService>(device, ipcFactory, sdk);
    endpoint.addService<SetupService>(device, ipcFactory, sdk);
    endpoint.addService<NetworkService>(ipcFactory);
    endpoint.addService<NotificationService>(device, ipcFactory, authProvider, stereoPairProvider, userConfigProvider, sdk);
    endpoint.addService<NtpService>(device, ipcFactory, userConfigProvider);
    endpoint.addService<BugReportService>(device, ipcFactory, userConfigProvider);

    addMediaServices(endpoint, device, ipcFactory, sdk);

    if (isStereoPairSupported) {
        endpoint.addService<StereoPairService>(
            device,
            ipcFactory,
            clockTowerProvider,
            glagolClusterProvider,
            spectrogramAnimationProvider,
            userConfigProvider,
            volumeManagerProvider,
            sdk);
    }

    {
        const auto audiodConfig = device->configuration()->getServiceConfig("audiod");
        endpoint.addService<AudioSourceService>("AudioSourceService", openAudioDevice(audioDeviceParam, audiodConfig), sdk, device, ipcFactory);
    }

    if (opts["use-glagol"].as<bool>()) {
        YIO_LOG_WARN("Warning! Glagol Service use avahi. Disable local avahi daemon to avoid collisions");
        auto neighbors = opts.count("neighbor") ? opts["neighbor"].as<std::vector<std::tuple<std::string, int>>>() : std::vector<std::tuple<std::string, int>>();
        endpoint.addService<GlagolServiceWrapper>(device,
                                                  ipcFactory,
                                                  authProvider,
                                                  deviceStateProvider,
                                                  multiroomProvider,
                                                  stereoPairProvider,
                                                  nsdMessagerFactory,
                                                  std::move(neighbors),
                                                  sdk);
    } else {
        endpoint.addStubService(ipcFactory, "glagold");
    }

    endpoint.addService<FakeWifiService>(ipcFactory, "wifid");
    endpoint.addStubService(ipcFactory, "monitord");
    endpoint.addStubService(ipcFactory, "updatesd");
    endpoint.addStubService(ipcFactory, "videod");

    endpoint.startAll();

    // Enable Setup greetings
    auto configurationModeHandler = std::make_shared<ConfigurationModeHandler>(sdk, volumeManager);
    configurationModeHandler->subscribeToSDK();

    auto defaultUpdateGreetingObserver = std::make_shared<DefaultOnConfigureDoneGreetingObserver>(sdk, YandexIO::DeviceModeObserver::ConfigurationSuccessState::HAS_UPDATE, StartingGreetingData::CONFIGURE_SUCCESS_UPDATE_WAV);
    defaultUpdateGreetingObserver->subscribeToSDK();

    auto successGreetingObserver = std::make_shared<DefaultOnConfigureDoneGreetingObserver>(sdk, YandexIO::DeviceModeObserver::ConfigurationSuccessState::NO_UPDATE, StartingGreetingData::CONFIGURE_SUCCESS_WAV);
    successGreetingObserver->subscribeToSDK();

    /* equalizer */
    auto equalizerController = std::make_shared<YandexIO::EqualizerController>(std::make_unique<quasar::NamedCallbackQueue>("EqualizerController"),
                                                                               Json::Value{}, sdk, device->telemetry(), createEqualizerDispatcherFactory(sdk));
    equalizerController->subscribeToBackendConfig();
    sdk->getEndpointStorage()->getLocalEndpoint()->addCapability(equalizerController);
    sdk->addBackendConfigObserver(equalizerController);
    sdk->addSDKStateObserver(equalizerController);

    // Personalization
    const auto audioSourceClient = YandexIO::createAudioSourceClient(ipcFactory);
    audioSourceClient->subscribeToChannels(YandexIO::RequestChannelType::ALL);
    audioSourceClient->start();
    const auto bioCapabilitySwitcher = std::make_shared<YandexIO::BioCapabilitySwitcher>(sdk, device, nullptr, sdk->getEndpointStorage()->getLocalEndpoint(), authProvider, audioSourceClient);
    sdk->addBackendConfigObserver(bioCapabilitySwitcher);

    // VolumeCapability
    const auto volumeCapability = std::make_shared<YandexIO::VolumeCapability>(volumeManager);
    sdk->getEndpointStorage()->getLocalEndpoint()->addCapability(volumeCapability);

    // convert xiva push to aliceRequests
    const auto xivaAliceRequester = YandexIO::XivaPushAliceRequester::install(*sdk);

    const auto spotterConfigurer = YandexIO::SpotterConfigurer::install(
        std::make_unique<quasar::NamedCallbackQueue>("SpotterConfigurer"), *sdk, device);

    /* Call SDK start method, so firstrund will run Init mode (if not configured) or broadcast configured state */
    sdk->allowInitConfigurationState();

    volumeManager->start();

    /* Hack to synchronize services internal IPCs. Should be called after start because
     * Firstrund starts connection to Authd only after receiving DeviceContext::onQuasarStart callback
     */
    endpoint.sync();
    YIO_LOG_INFO("All endpoints started");

    if (config->has_auth_code()) {
        YIO_LOG_INFO("Authenticate using provided xCode");
        sdk->authenticate(config->auth_code());
    }
    KeyboardManager userInputManager(sdk);

    const bool useKeyInput = !opts["disable-key-input"].as<bool>();
    if (useKeyInput) {
        // block until "x" key input
        userInputManager.run();
    } else {
        // block until sigterm
        waiter.wait();
    }

    YIO_LOG_INFO("SampleApp exit...");

    return 0;
}
