#include "endpoint.h"

#include <yandex_io/libs/base/quasar_service.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <list>
#include <unordered_map>
#include <unordered_set>

#include <pthread.h>

using namespace quasar;

// This is a no-op implementation for services that always accepts
// connections, but does nothing. Incoming messages are logged.
// Providing a mock implementation prevents "cannot connect" errors in
// actual services.
class StubService: public QuasarService {
    std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
    std::string serviceName_;
    std::shared_ptr<ipc::IServer> server_;

public:
    explicit StubService(std::shared_ptr<ipc::IIpcFactory> ipcFactory, const std::string& name)
        : ipcFactory_(std::move(ipcFactory))
        , serviceName_(name)
    {
    }

    std::string getServiceName() const override {
        return serviceName_;
    }

    void start() override {
        server_ = ipcFactory_->createIpcServer(serviceName_);
        server_->setMessageHandler([this](const ipc::SharedMessage& message, auto& /* handler */) {
            YIO_LOG_DEBUG("STUB: Unhandled message for " << serviceName_ << ": " << shortUtf8DebugString(*message));
        });
        server_->listenService();
        YIO_LOG_INFO("Start stub service \"" << serviceName_ << "\" on port: " << server_->port());
    }
};

YandexIOEndpoint::YandexIOEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::string endpointName)
    : device_(std::move(device))
    , endpointName_(std::move(endpointName))
{
}

void YandexIOEndpoint::startAll()
{
    const auto excludedServicesJson = device_->configuration()->getServiceConfig(endpointName_)["excludedServices"];
    std::unordered_set<std::string> excludedServices;

    YIO_LOG_INFO("Excluded services: " << excludedServicesJson);
    for (const Json::Value& serviceJson : excludedServicesJson)
    {
        excludedServices.insert(serviceJson.asString());
    }

    std::list<std::thread> servicesStartingThreads;
    for (auto& service : services_) {
        if (!excludedServices.count(service->getServiceName()))
        {
            YIO_LOG_INFO("Starting " << service->getServiceName() << " Service");
            servicesStartingThreads.emplace_back(&QuasarService::start, service.get());
            pthread_setname_np(servicesStartingThreads.back().native_handle(), service->getThreadName().c_str());
        }
    }
    for (auto& thread : servicesStartingThreads) {
        thread.join();
    }
    YIO_LOG_INFO("Services successfully started");
}

void YandexIOEndpoint::sync()
{
    for (auto& service : services_) {
        service->sync();
    }
}

void YandexIOEndpoint::addService(std::unique_ptr<QuasarService> service)
{
    services_.push_back(std::move(service));
}

void YandexIOEndpoint::addStubService(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, const std::string& name) {
    addService<StubService>(ipcFactory, name);
}
