#include "entry.h"

#include <fstream>
#include <yandex_io/scaffolding/cli_tool/cli.h>
#include <yandex_io/scaffolding/telemetry/client/telemetry.h>

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/mixed/mixed_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/net_test/net_test.h>
#include <yandex_io/libs/net/wifi_info.h>

#include <build/scripts/c_templates/svnversion.h>

#include <util/folder/path.h>
#include <util/system/yassert.h>

#include <cstdio>
#include <cstdlib>
#include <iostream>

namespace {

    int printVersion(QuasarCallParams& params) {
        std::cout << "\nSoftware Revision: " << GetProgramCommitId() << std::endl;
        std::cout << "Quasar Software Version: " << params.device()->softwareVersion() << std::endl;
        return EXIT_SUCCESS;
    }

    int printBuildTimestamp(QuasarCallParams& /*params*/) {
        std::cout << "\nBuild Timestamp: " << GetProgramBuildTimestamp() << std::endl;
        return EXIT_SUCCESS;
    }

    int printDeviceID(QuasarCallParams& params) {
        std::cout << "\nDevice ID: " << params.device()->deviceId() << std::endl;
        return EXIT_SUCCESS;
    }

    int printFullInfo(QuasarCallParams& params) {
        printVersion(params);
        printBuildTimestamp(params);
        printDeviceID(params);

        return EXIT_SUCCESS;
    }

    int setDevBackendFlag(QuasarCallParams& params) {
        // create flag
        quasar::PersistentFile flag(params.device()->configuration()->getDevBackendFlagPath(), quasar::PersistentFile::Mode::TRUNCATE);
        return EXIT_SUCCESS;
    }

    int removeDevBackendFlag(QuasarCallParams& params) {
        std::remove(params.device()->configuration()->getDevBackendFlagPath().c_str());
        return EXIT_SUCCESS;
    }

    int cli(QuasarCallParams& params) {
        return RunYandexIOSDKCli(params.mixedIpcFactory());
    }

    int copyLastMonotonicTime(QuasarCallParams& params) {
        const auto configuration = params.device()->configuration();
        const auto monotonicClockFile = quasar::tryGetString(configuration->getServiceConfig("ntpd"), "monotonicClockFile");
        const auto lastMonotonicTimeFile = quasar::tryGetString(configuration->getServiceConfig("maind"), "lastMonotonicClockFile");

        TFsPath original(monotonicClockFile);
        if (original.Exists()) {
            constexpr bool force = true; // will create dirs if not exists
            original.CopyTo(TString(lastMonotonicTimeFile), force);
        }
        return 0;
    }

    struct Pretty {
        uint64_t b;
    };
    std::ostream& operator<<(std::ostream& o, const Pretty& p) {
        const char prefixes[] = "KMGTPE"; // 2^64 is 16 exabytes
        int idx = 0;
        auto b = p.b;
        while (b > 1024) {
            b /= 1024;
            ++idx;
        };
        o << b;
        if (idx) {
            o << prefixes[idx];
        }
        return o;
    }

    int nettest(QuasarCallParams& params) {
        if (params.argc() < 3) {
            std::cout << "Usage: --nettest URL_TO_DOWNLOAD BROADCASTS_PER_SECOND\n";
            return 0;
        }

        std::string url = params.arg(2);
        uint64_t bps = std::atol(params.arg(3));
        utility::testNetwork(url, bps, [](const utility::NetTestStats& stats) {
            std::cout << "Downloaded " << stats.bytesLoaded
                      << " at speed " << Pretty{stats.loadSpeed} << "B/s."
                      << " Broadcasted " << stats.broadcastsSent
                      << " at " << stats.sentRate << " P/s" << std::endl;
        });

        return EXIT_SUCCESS;
    }

    int wifiInfo(QuasarCallParams& params) {
        const std::string wlanName = params.argc() > 2 ? params.arg(2) : "wlan0";
        std::cout << "Detailed info about device '" << wlanName << "'" << std::endl;
        const quasar::net::WifiInfoSettings settings{
            .allBss = true,
            .iface = true,
            .tids = true,
            .bothIE = true,
            .longCpblt = true,
            .survey = true,
        };
        Json::Value result = quasar::net::getDetailedWifiInfo(wlanName, settings);
        std::cout << result << '\n';
        return EXIT_SUCCESS;
    }

    void printSchemeOptions(const QuasarCallScheme& scheme) {
        bool first = true;
        for (const auto& item : scheme) {
            if (!first) {
                std::cout << ", ";
            }
            std::cout << item.first;
            first = false;
        }
    }

    std::string getOsType(const std::string& model) {
        if (model == "yandexstation_2" || model == "yandexstation") {
            return "Android";
        } else {
            return "Linux";
        }
    }

    int getprop(QuasarCallParams& params) {
        const auto& device = params.device();
        const auto& model = device->platform();
        Json::Value props = Json::objectValue;
        props["ro.product.model"] = model;
        props["ro.product.brand"] = model;
        props["ro.product.manufacturer"] = model;
        props["gsm.sim.operator.alpha"] = "";
        props["gsm.operator.alpha"] = "";
        props["ro.product.name"] = model;
        props["ro.board.platform"] = getOsType(model);
        props["ro.opengles.version"] = "1";
        props["ro.product.device"] = model;

        props["ro.product.cpu.abi"] = "arm64-v8a";
        props["ro.product.cpu.abilist"] = "arm64-v8a,armeabi-v7a,armeabi";
        props["ro.product.cpu.abilist32"] = "armeabi-v7a,armeabi";
        props["ro.product.cpu.abilist64"] = "arm64-v8a";

        props["ro.build.version.sdk"] = device->softwareVersion();
        props["ro.build.version.preview_sdk"] = "0";
        props["ro.build.version.release"] = device->softwareVersion();

        props["sys.boot_completed"] = "1";

        if (params.argc() > 2) {
            std::ofstream propfile{params.arg(2)};
            propfile << quasar::jsonToString(props);
        } else {
            std::cout << quasar::jsonToString(props);
        }

        return 0;
    }

    int help(const DaemonsCallScheme& daemons, const QuasarCallScheme& tools, const char* executable) {
        std::cout << "Usage: " << executable << " <OPTION>" << std::endl;
        std::cout << "Tools name options: ";
        printSchemeOptions(tools);
        std::cout << std::endl;

        std::cout << "Daemons name options: ";
        printSchemeOptions(daemons);
        std::cout << std::endl;

        return EXIT_FAILURE;
    }

    void setTelemetrySenderName(QuasarCallParams& params, const std::string& name) {
        auto makeSenderName = [](const auto& name) -> std::string {
            return name.starts_with("--") ? name.substr(2) : name;
        };
        params.device()->telemetry()->setSenderName(makeSenderName(name));
    }
} // anonymous namespace

int callDaemons(QuasarCallParams& params, const DaemonsCallScheme& daemons) {
    /* Tools to get information about software */
    QuasarCallScheme tools = {
        {"--version", printVersion},
        {"--build-timestamp", printBuildTimestamp},
        {"--device-id", printDeviceID},
        {"--full-info", printFullInfo},
        {"--set-dev-backend-flag", setDevBackendFlag},
        {"--remove-dev-backend-flag", removeDevBackendFlag},
        {"--cli", cli},
        {"--nettest", nettest},
        {"--wifi-info", wifiInfo},
        {"--save-last-monotonic-time", copyLastMonotonicTime},
        {"--getprop", getprop},
    };

    if (params.argc() >= 2) {
        /**
         * Init Curl before any thread run. It is necessary because curl_global_init is NOT THREAD SAFE
         */
        quasar::HttpClient::globalInit();

        /* Run daemons */
        const std::string selector = params.arg(1);
        if (auto it = daemons.find(selector); it != daemons.end()) {
            setTelemetrySenderName(params, selector);
            auto& quasarMain = it->second;
            return quasarMain(params);
        }

        if (auto it = tools.find(selector); it != tools.end()) {
            setTelemetrySenderName(params, selector);
            auto& quasarMain = it->second;
            auto res = quasarMain(params);
            // After completing the main work of console command, we increase
            // the logging level so as not to read technical spam
            quasar::Logging::changeLoggerLevel("warn");
            return res;
        }
    }

    return help(daemons, tools, params.arg(0));
}

QuasarCallParams::QuasarCallParams(Args args)
    : args_(std::move(args))
{
}

QuasarCallParams::~QuasarCallParams() {
    // device_ is a complex object that contains telemetry references
    // and should therefore be removed first
    device_.reset();
}

const char* QuasarCallParams::arg(int index) const {
    Y_VERIFY(index < args_.argc);
    return args_.argv[index];
}

std::shared_ptr<YandexIO::Configuration> QuasarCallParams::configuration() {
    auto lock = std::scoped_lock{mutex_};

    if (!configuration_) {
        configuration_ = YandexIO::makeConfiguration();
    }

    return configuration_;
}

std::shared_ptr<YandexIO::IDevice> QuasarCallParams::device() {
    auto lock = std::scoped_lock{mutex_};

    if (!device_) {
        auto telemetry = makeTelemetry(mixedIpcFactory());
        device_ = std::make_shared<YandexIO::Device>(
            std::move(args_.deviceId),
            configuration(),
            std::move(telemetry),
            std::move(args_.hal),
            std::move(args_.color),
            std::move(args_.revision));
    }

    return device_;
}

std::shared_ptr<quasar::ipc::MixedIpcFactory> QuasarCallParams::mixedIpcFactory() {
    auto lock = std::scoped_lock{mutex_};

    if (!mixedIpcFactory_) {
        auto context = quasar::ipc::MixedIpcFactory::createDefaultContext(configuration());
        mixedIpcFactory_ = std::make_shared<quasar::ipc::MixedIpcFactory>(std::move(context));
    }

    return mixedIpcFactory_;
}
