#include "calld.h"

#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/callkit/cli/cli.h>
#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/logging_handler/logging_handler.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/telemetry_handler/telemetry_handler.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/services/calld/call_connector.h>
#include <yandex_io/services/calld/call_service.h>
#include <yandex_io/sdk/yandex_iosdk.h>

#include <util/generic/size_literals.h>

#include <memory>

using namespace quasar;

namespace {
    constexpr const char* PROCESS_NAME = "calld";
} // namespace

int calld(QuasarCallParams& params) {
    TerminateWaiter waiter;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    ipcFactory->setProcessName(PROCESS_NAME);
    sendDaemonStartMetric(PROCESS_NAME, device);

    auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, PROCESS_NAME, device->deviceId());

    LoggingHandler loggingHandler;
    loggingHandler.init(ipcFactory, CallService::SERVICE_NAME, device->configuration()->getFullConfig(), device->telemetry());

    TelemetryHandler telemetryHandler;
    telemetryHandler.init(device->telemetry(), ipcFactory);

    auto config = device->configuration()->getServiceConfig(CallService::SERVICE_NAME);
    QuasarMinidump::getInstance().init(CallService::SERVICE_NAME, config, device->telemetry(), device->softwareVersion());

    auto authProvider = std::make_shared<AuthProvider>(ipcFactory);
    auto stereoPairProvider = std::make_shared<StereoPairProvider>(device, ipcFactory);

    CallService callService(device, ipcFactory, authProvider, stereoPairProvider, std::move(sdk));
    callService.start();

    waiter.wait();
    return 0;
}

int call_util(QuasarCallParams& params)
{
    quasar::Logging::initLoggingToFile("call_util.log", "debug", 10_KB, 1);

    const auto ipcFactory = params.mixedIpcFactory();
    ipcFactory->setProcessName("call_util");
    return messenger::cli(std::make_unique<quasar::CallConnector>(ipcFactory), params.argc(), params.argv());
}
