#include "glagol_avahi_server_watchdog.h"

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/avahi_wrapper/avahi_browse_client.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("glagol");

using namespace quasar;
using namespace glagol;
using namespace std::chrono;

GlagolAvahiServerWatchdog::GlagolAvahiServerWatchdog(std::shared_ptr<YandexIO::IDevice> device, bool shouldAbortIfServerLost, std::shared_ptr<ResponderStatus> responderStatus)
    : device_(std::move(device))
    , responderStatus_(std::move(responderStatus))
    , shouldAbortIfServerLost_(shouldAbortIfServerLost)
{
    deviceId_.id = device_->deviceId();
    deviceId_.platform = device_->configuration()->getDeviceType();
    lastVisibilityChange_ = std::chrono::steady_clock::now();
    maxServerNonVisiblePeriod_ = std::chrono::seconds(tryGetInt(device_->configuration()->getServiceConfig("glagold"), "maxServerNonVisiblePeriodSec", 31));

    discovery_ = std::make_shared<Discovery>(
        nullptr,
        std::make_shared<AvahiBrowseClient>(device_->deviceId()),
        Discovery::Settings(),
        std::bind(&GlagolAvahiServerWatchdog::onDiscoveryResultChanged, this, std::placeholders::_1));

    periodicExecutor_ = std::make_unique<PeriodicExecutor>(
        std::bind(&GlagolAvahiServerWatchdog::checkServerVisibility, this),
        maxServerNonVisiblePeriod_ / 2);
}

void GlagolAvahiServerWatchdog::onDiscoveryResultChanged(const glagol::Discovery::Result& result) {
    std::lock_guard<std::mutex> lock(mutex_);
    int visibleNew = result.items.count(deviceId_);
    if (visible_ != visibleNew) {
        visible_ = visibleNew;
        lastVisibilityChange_ = std::chrono::steady_clock::now();
        YIO_LOG_INFO("Avahi server visibility changed! " << visible_);
    }
}

void GlagolAvahiServerWatchdog::checkServerVisibility() {
    std::lock_guard<std::mutex> lock(mutex_);
    YIO_LOG_DEBUG("Avahi checking server visibility");
    const auto timeDelta = steady_clock::now() - lastVisibilityChange_;
    if (!visible_ && timeDelta > maxServerNonVisiblePeriod_) {
        if (responderStatus_->isNetworkAvailable()) {
            YIO_LOG_ERROR_EVENT("GlagolAvahiServerWatchdog.MissingAvahiServer", "Avahi server is not visible for " << duration_cast<seconds>(timeDelta).count() << " seconds. Restarting glagold");
            device_->telemetry()->reportError("GlagolAvahiServerLost", responderStatus_->getResponderStatusJson());
            std::this_thread::sleep_for(seconds(5));
            if (shouldAbortIfServerLost_) {
                kill(getpid(), SIGKILL); // FIXME do not abort when glagold moves to yiod
            }
        } else {
            YIO_LOG_INFO("Avahi server is not visible for " << duration_cast<seconds>(timeDelta).count() << " seconds. No network.");
        }
    }
}
