#include "glagold.h"

#include "glagol_avahi_server_watchdog.h"

#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/multiroom/connector/multiroom_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging_handler/logging_handler.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/telemetry_handler/telemetry_handler.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/services/glagold/glagol.h>
#include <yandex_io/services/glagold/mdnsd_nsd_messager.h>
#include <yandex_io/sdk/yandex_iosdk.h>

#include <chrono>
#include <csignal>
#include <thread>

using namespace quasar;

namespace {

    constexpr const char* PROCESS_NAME = "glagold";

} // namespace

int glagold(QuasarCallParams& params)
{
    TerminateWaiter waiter;

    quasar::SteadyConditionVariable condvar;
    std::mutex glagoldMutex;
    Glagol::Settings glagolSettings;
    bool shouldRunWatchDog = false;
    bool shouldAbortIfServerLost = false;
    bool configInitialized = false;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    ipcFactory->setProcessName(PROCESS_NAME);
    sendDaemonStartMetric(PROCESS_NAME, device);

    LoggingHandler loggingHandler;
    loggingHandler.init(ipcFactory, GLAGOLD_SERVICE_NAME, device->configuration()->getFullConfig(), device->telemetry());

    TelemetryHandler telemetryHandler;
    telemetryHandler.init(device->telemetry(), ipcFactory);

    const auto& config = device->configuration()->getServiceConfig(GLAGOLD_SERVICE_NAME);
    QuasarMinidump::getInstance().init(GLAGOLD_SERVICE_NAME, config, device->telemetry(), device->softwareVersion());

    {
        auto syncdConnector = ipcFactory->createIpcConnector("syncd");
        syncdConnector->setMessageHandler([&](const auto& message) {
            if (message->has_user_config_update()) {
                std::lock_guard<std::mutex> lock(glagoldMutex);

                try {
                    auto systemConfig = parseJson(message->user_config_update().config())["system_config"];

                    auto settings = Glagol::fromSystemConfig(systemConfig);
                    auto runWatchDog = tryGetBool(systemConfig, "avahiWatchdog", false);
                    auto shouldAbort = tryGetBool(systemConfig, "avahiWatchdogAbortIfServerLost", false);

                    if (!configInitialized) {
                        // first-time entry
                        glagolSettings = settings;
                        shouldRunWatchDog = runWatchDog;
                        shouldAbortIfServerLost = shouldAbort;
                        configInitialized = true;
                        YIO_LOG_DEBUG("Received config, restricted {" << glagolSettings.avahi.restrictToIPv4
                                                                      << "}, nsdInsteadOfAvahi {" << glagolSettings.nsdInsteadOfAvahi
                                                                      << "}, heartbeatTelemetry {" << glagolSettings.heartbeatTelemetry
                                                                      << "}");

                        condvar.notify_one();
                    }
                } catch (Json::Exception& e) {
                    YIO_LOG_WARN("Failed to parse json from systemConfig: {" << e.what() << "}");
                }
            }
        });

        syncdConnector->connectToService();

        // wait for message handler to receive config once
        std::unique_lock<std::mutex> lock(glagoldMutex);
        condvar.wait(lock, [&] { return configInitialized; });
    }

    YIO_LOG_INFO("Syncd interaction completed");
    YIO_LOG_INFO("Starting service " << GLAGOLD_SERVICE_NAME << " version " << device->softwareVersion());

    const bool isMultiroomSupported = device->configuration()->isFeatureSupported("multiroom");
    const bool isStereoPairSupported = device->configuration()->isFeatureSupported("stereo_pair");

    MdnsdMessagerFactory nsdMessagerFactory{ipcFactory};

    auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, PROCESS_NAME, device->deviceId());

    const auto glagold = std::make_shared<Glagol>(
        device,
        ipcFactory,
        std::make_shared<AuthProvider>(ipcFactory),
        std::make_shared<DeviceStateProvider>(ipcFactory),
        (isMultiroomSupported ? std::make_shared<MultiroomProvider>(device, ipcFactory) : nullptr),
        (isStereoPairSupported ? std::make_shared<StereoPairProvider>(device, ipcFactory) : nullptr),
        createDefaultMndsFactory(nsdMessagerFactory),
        config,
        glagolSettings,
        sdk);
    sdk->getEndpointStorage()->getLocalEndpoint()->addListener(glagold);

    std::unique_ptr<GlagolAvahiServerWatchdog> glagolAvahiServerWatchdog;

    if (shouldRunWatchDog) {
        YIO_LOG_INFO("Starting GlagolAvahiServerWatchdog, wait for glagold WS-server");
        glagold->waitWsServerStart();
        YIO_LOG_INFO("WS-server started, run watchdog");
        glagolAvahiServerWatchdog = std::make_unique<GlagolAvahiServerWatchdog>(device, shouldAbortIfServerLost, glagold->getResponderStatusProvider());
    }

    waiter.wait();
    YIO_LOG_INFO("Terminated");

    return 0;
}
