#include "mediad.h"

#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/clock_tower/connector/clock_tower_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/glagol/connector/glagol_cluster_provider.h>
#include <yandex_io/interfaces/multiroom/connector/multiroom_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/libs/audio_player/base/audio_clock.h>
#include <yandex_io/libs/audio_player/factory/player_factory_selector.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging_handler/logging_handler.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/telemetry_handler/telemetry_handler.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/services/mediad/audio_clock_manager/audio_clock_manager.h>
#include <yandex_io/services/mediad/media_service.h>
#include <yandex_io/modules/testpoint/testpoint_peer.h>
#include <yandex_io/sdk/yandex_iosdk.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("media");

using namespace quasar;
using namespace quasar::proto;

namespace {
    class MediaServicesWrapper: public QuasarService {
    public:
        MediaServicesWrapper(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory, std::shared_ptr<YandexIO::SDKInterface> sdk)
            : device_(std::move(device))
            , ipcFactory_(std::move(ipcFactory))
            , sdk_(std::move(sdk))
        {
        }
        std::string getServiceName() const override {
            return "mediad";
        }
        void start() override {
            const auto& mediadConfig = device_->configuration()->getServiceConfig(MediaEndpoint::SERVICE_NAME);
            const auto& netClockConfig = device_->configuration()->getServiceConfig("net_clock");
            const int localAudioClockPort = tryGetInt(netClockConfig, "port", 0);
            const auto audioClockManager = std::make_shared<AudioClockManager>(device_->deviceId(), localAudioClockPort, AudioClockManager::generateClockId());

            const bool isMultiroomSupported = device_->configuration()->isFeatureSupported("multiroom");
            const bool isStereoPairSupported = device_->configuration()->isFeatureSupported("stereo_pair");

            auto authProvider = std::make_shared<AuthProvider>(ipcFactory_);
            auto clockTowerProvider = std::make_shared<ClockTowerProvider>(ipcFactory_);
            auto deviceStateProvider = std::make_shared<DeviceStateProvider>(ipcFactory_);
            auto glagolClusterProvider = std::make_shared<GlagolClusterProvider>(ipcFactory_);
            auto multiroomProvider = (isMultiroomSupported ? std::make_shared<MultiroomProvider>(device_, ipcFactory_) : nullptr);
            auto stereoPairProvider = (isStereoPairSupported ? std::make_shared<StereoPairProvider>(device_, ipcFactory_) : nullptr);
            auto userConfigProvider = std::make_shared<UserConfigProvider>(ipcFactory_);
            auto audioPlayerFactory = std::make_shared<PlayerFactorySelector>(mediadConfig);
            auto mediaPlayerFactory = std::make_shared<PlayerFactory>(device_, ipcFactory_, audioClockManager, audioPlayerFactory, sdk_);
            mediaPlayerFactory->addYandexMusicConstructor(true, multiroomProvider, stereoPairProvider, userConfigProvider);
            mediaPlayerFactory->addYandexRadioConstructor();

            mediaService_ = std::make_unique<MediaService>(device_, ipcFactory_, authProvider, deviceStateProvider, multiroomProvider, stereoPairProvider, mediaPlayerFactory, sdk_);
            mediaService_->start();

            testpoint_ = std::make_unique<YandexIO::TestpointPeer>(*ipcFactory_);
        }

    private:
        std::unique_ptr<MediaService> mediaService_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory_;
        std::unique_ptr<YandexIO::TestpointPeer> testpoint_;
        const std::shared_ptr<YandexIO::SDKInterface> sdk_;
    };

    constexpr const char* PROCESS_NAME = "mediad";
} // namespace

int mediad(QuasarCallParams& params)
{
    TerminateWaiter waiter;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    ipcFactory->setProcessName(PROCESS_NAME);
    sendDaemonStartMetric(PROCESS_NAME, device);

    LoggingHandler loggingHandler;
    loggingHandler.init(ipcFactory, MediaEndpoint::SERVICE_NAME, device->configuration()->getFullConfig(), device->telemetry());

    TelemetryHandler telemetryHandler;
    telemetryHandler.init(device->telemetry(), ipcFactory);

    YIO_LOG_INFO("Starting service mediad version " << device->softwareVersion());

    const auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, PROCESS_NAME, device->deviceId());

    const auto& mediadConfig = device->configuration()->getServiceConfig(MediaEndpoint::SERVICE_NAME);
    QuasarMinidump::getInstance().init(MediaEndpoint::SERVICE_NAME, mediadConfig, device->telemetry(), device->softwareVersion());

    MediaServicesWrapper mediaServicesWrapper(device, ipcFactory, sdk);
    mediaServicesWrapper.start();
    waiter.wait();

    return 0;
}

void addMediaServices(quasar::YandexIOEndpoint& endpoint, std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory, std::shared_ptr<YandexIO::SDKInterface> sdk) {
    endpoint.addService<MediaServicesWrapper>(std::move(device), std::move(ipcFactory), std::move(sdk));
}
