#include "scaffolding.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>

#include <boost/algorithm/string.hpp>

#include <util/folder/path.h>
#include <util/system/env.h>

#include <memory>

#include <sys/resource.h>
#include <sys/time.h>

using namespace quasar;

void prepareEnvironment(const LauncherConfig& config) {
    if (config.has_paths()) {
        const auto& paths = config.paths();
        if (paths.has_workdir_path()) {
            TFsPath workdir(paths.workdir_path());
            workdir.MkDirs();
            if (chdir(workdir.GetPath().c_str()) != 0) {
                throw std::runtime_error(std::string("Failed to change working directory") + strerror(errno));
            }
        }
    }

    if (config.has_limits()) {
        const auto& limits = config.limits();
        if (limits.has_fileno_limit()) {
            struct rlimit limit;
            if (getrlimit(RLIMIT_NOFILE, &limit) != 0) {
                throw std::runtime_error(std::string("Failed to get fileno rlimit: ") + strerror(errno));
            }

            limit.rlim_cur = limits.fileno_limit();
            if (setrlimit(RLIMIT_NOFILE, &limit) != 0) {
                throw std::runtime_error(std::string("Failed to set fileno rlimit: ") + strerror(errno));
            }
        }
    }

    if (config.has_ca_certificates_path()) {
        // provide ca-cert path for gstreamer: https://a.yandex-team.ru/arc/trunk/arcadia/yandex_io/libs/audio_player/gstreamer/gstreamer_audio_player.cpp?rev=r8217878#L225-243
        SetEnv(TString("CA_CERTIFICATES"), TString(config.ca_certificates_path()));
    }

    // XXX: Bypass firstrund init sequence by transferring it to the final state
    // by flag file
    PersistentFile setRegistered("registered", PersistentFile::Mode::TRUNCATE);
}

YandexIO::DeviceID makeDeviceId(YandexIO::Configuration* configuration) {
    const auto config = configuration->getServiceConfig("common");
    const auto deviceIdFileName = config["deviceIdFileName"];

    if (deviceIdFileName.isString()) {
        return boost::trim_copy(getFileContent(deviceIdFileName.asString()));
    } else {
        throw std::runtime_error("Invalid deviceIdFileName");
    }
}
