#include "telemetry.h"

#include <yandex_io/scaffolding/base/endpoint.h>
#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging_handler/logging_handler.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/self_destroyer/self_destroyer_utils.h>
#include <yandex_io/libs/telemetry_handler/telemetry_handler.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/metrica/metrica2d/metrica2_service.h>
#include <yandex_io/metrica/metricad/metrica_service.h>
#include <yandex_io/metrica/teleme3d/teleme3d_service.h>
#include <yandex_io/metrica/monitor/monitor_service.h>
#include <yandex_io/services/telemetry_proxy/telemetry_proxy_service.h>

#include <memory>

using namespace quasar;

namespace {
    constexpr const char* ENDPOINT_NAME = "telemetry";

    bool teleme3dEnabled(const std::shared_ptr<YandexIO::IDevice>& device) {
        const auto& serviceConfig = device->configuration()->getServiceConfig("telemetry");
        return serviceConfig["enableTeleme3d"].asBool();
    }
} // namespace

int telemetry(QuasarCallParams& params, std::unique_ptr<quasar::IMetricsCollector> collector)
{
    TerminateWaiter waiter;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    sendDaemonStartMetric("telemetry", device);

    LoggingHandler loggingHandler;
    {
        loggingHandler.init(ipcFactory, ENDPOINT_NAME, device->configuration()->getFullConfig(), device->telemetry());

        const auto& config = device->configuration()->getServiceConfig(ENDPOINT_NAME);
        QuasarMinidump::getInstance().init(ENDPOINT_NAME, config, device->telemetry(), device->softwareVersion());
    }

    TelemetryHandler telemetryHandler;
    telemetryHandler.init(device->telemetry(), ipcFactory);

    YIO_LOG_INFO("Starting service Telemetry version " << device->softwareVersion());

    YandexIOEndpoint telemetryEndpoint(device, ENDPOINT_NAME);

    addTelemetryServices(telemetryEndpoint, params, std::move(collector));
    telemetryEndpoint.startAll();

    const auto destroyer = SelfDestroyerUtils::create(
        "telemetry",
        device,
        "telemetry");
    quasar::PeriodicExecutor killer([&]() {
        destroyer->destroySelfIfNeed();
    }, std::chrono::hours(1));

    waiter.wait();

    return 0;
}

void addTelemetryServices(quasar::YandexIOEndpoint& endpoint, QuasarCallParams& params, std::unique_ptr<quasar::IMetricsCollector> collector) {
    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    auto authProvider = std::make_shared<AuthProvider>(ipcFactory);
    if (teleme3dEnabled(device)) {
        endpoint.addService<Teleme3dService>(device, ipcFactory, authProvider);
    } else {
        endpoint.addService<MetricaService>(device, ipcFactory, authProvider);
        endpoint.addService<Metrica2Service>(device, ipcFactory);
    }
    endpoint.addService<MonitorService>(device, ipcFactory, std::move(collector));
    endpoint.addService<TelemetryProxyService>(device, ipcFactory);
}
