#include "yiod.h"

#include <yandex_io/scaffolding/base/endpoint.h>
#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/sdk/hub/io_hub_service.h>
#include <yandex_io/sdk/yandex_iosdk.h>
#include <yandex_io/services/adbd/adb_service.h>
#include <yandex_io/services/alarmd/alarm_service.h>
#include <yandex_io/services/aliced/alice_service.h>
#include <yandex_io/services/braind/brain_service.h>
#include <yandex_io/services/brickd/brick_service.h>
#include <yandex_io/services/bug_report/bug_report_service.h>
#include <yandex_io/services/do_not_disturb/do_not_disturb_service.h>
#include <yandex_io/services/iot/iot_service.h>
#include <yandex_io/services/networkd/network_service.h>
#include <yandex_io/services/notificationd/notification_service.h>
#include <yandex_io/services/ntpd/ntp_service.h>
#include <yandex_io/services/pushd/push_service.h>
#include <yandex_io/services/stereo_paird/stereo_pair_service.h>
#include <yandex_io/services/syncd/sync_service.h>

#if !defined(__ANDROID__)

    #include <yandex_io/services/ble_initd/ble_service.h>

#endif

#include <yandex_io/services/setupd/setup_service.h>
#include <yandex_io/services/sound_initd/sound_init_service.h>

#include <yandex_io/services/authd/auth_service.h>
#include <yandex_io/services/firstrund/first_run_service.h>

#include <yandex_io/services/audiosender/audio_sender_service.h>
#include <yandex_io/services/audiosender/info/info.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/clock_tower/connector/clock_tower_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/glagol/connector/glagol_cluster_provider.h>
#include <yandex_io/interfaces/multiroom/connector/multiroom_provider.h>
#include <yandex_io/interfaces/spectrogram_animation/connector/spectrogram_animation_provider.h>
#include <yandex_io/interfaces/stereo_pair/connector/stereo_pair_provider.h>
#include <yandex_io/interfaces/updates/connector/updates_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/interfaces/volume_manager/connector/volume_manager_provider.h>

#include <yandex_io/sdk/yandex_iosdk.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging_handler/logging_handler.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/telemetry_handler/telemetry_handler.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/self_destroyer/self_destroyer_utils.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/speechkit_audio_player/devnull_audio_player.h>

#include <speechkit/PlatformTypes.h>

#include <memory>
#include <unistd.h>

using namespace quasar;

namespace {
    constexpr const char* ENDPOINT_NAME = "yiod";

    SpeechKit::AudioPlayer::SharedPtr createAudioPlayer(const Json::Value& config) {
        const auto playerString = tryGetString(config, "audioPlayer", "speechkit");
        SpeechKit::AudioPlayer::SharedPtr player;
        if (playerString == "speechkit") {
            const bool closeOpenALWhenIdle = tryGetBool(config, "closeOpenALWhenIdle", false);
            player = ::SpeechKit::PlatformTypes::createDefaultAudioPlayer(closeOpenALWhenIdle);
        } else if (playerString == "devnull") {
            player = DevnullAudioPlayer::create();
        } else {
            throw std::logic_error("Unknown aliced player type " + playerString);
        }
        return player;
    }

    class UpdaterSwitcherFlagService: public QuasarService {
        std::shared_ptr<IUserConfigProvider> configProvider_;
        Lifetime lifetime_;
        static constexpr const char* DEFAULT_UPDATESD_TOGGLE_PATH = "/data/quasar/use_updater";

    public:
        explicit UpdaterSwitcherFlagService(std::shared_ptr<IUserConfigProvider> provider)
            : configProvider_(std::move(provider))
                  {};

        ~UpdaterSwitcherFlagService() {
            lifetime_.die();
        };

        std::string getServiceName() const override {
            return "updater_switcher_flag_service";
        }

        void start() override {
            const auto serviceName = getServiceName();
            configProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "use_updater").connect([](const auto& jsonPtr) {
                if (!jsonPtr->isNull() && jsonPtr->asBool()) {
                    PersistentFile(DEFAULT_UPDATESD_TOGGLE_PATH, quasar::PersistentFile::Mode::TRUNCATE);
                } else {
                    std::remove(DEFAULT_UPDATESD_TOGGLE_PATH);
                };
            }, lifetime_);
        }
    };

    constexpr const char* PROCESS_NAME = "yiod";
} // namespace

int yiod(QuasarCallParams& params)
{
    TerminateWaiter waiter;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    ipcFactory->setProcessName(PROCESS_NAME);
    sendDaemonStartMetric(PROCESS_NAME, device);

    LoggingHandler loggingHandler;
    loggingHandler.init(ipcFactory, ENDPOINT_NAME, device->configuration()->getFullConfig(), device->telemetry());

    TelemetryHandler telemetryHandler;
    telemetryHandler.init(device->telemetry(), ipcFactory);

    YIO_LOG_INFO("Starting service yiod version " << device->softwareVersion());

    YandexIOEndpoint yandexIOEndpoint(device, ENDPOINT_NAME);

    const auto& yiodConfig = device->configuration()->getServiceConfig(ENDPOINT_NAME);
    QuasarMinidump::getInstance().init(ENDPOINT_NAME, yiodConfig, device->telemetry(), device->softwareVersion());

    auto selfDestroyer = addYiodServices(yandexIOEndpoint, params);

    yandexIOEndpoint.startAll();

    quasar::PeriodicExecutor killer([&]() {
        selfDestroyer->destroySelfIfNeed();
    }, std::chrono::minutes(30));

    waiter.wait();

    _exit(0);

    return 0;
}

std::shared_ptr<quasar::SelfDestroyer> addYiodServices(quasar::YandexIOEndpoint& yandexIOEndpoint, QuasarCallParams& params) {
    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();

    const auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, PROCESS_NAME, device->deviceId());

    const bool isMultiroomSupported = device->configuration()->isFeatureSupported("multiroom");
    const bool isStereoPairSupported = device->configuration()->isFeatureSupported("stereo_pair");

    auto authProvider = std::make_shared<AuthProvider>(ipcFactory);
    auto clockTowerProvider = (isMultiroomSupported ? std::make_shared<ClockTowerProvider>(ipcFactory) : nullptr);
    auto deviceStateProvider = std::make_shared<DeviceStateProvider>(ipcFactory);
    auto glagolClusterProvider = std::make_shared<GlagolClusterProvider>(ipcFactory);
    auto multiroomProvider = (isMultiroomSupported ? std::make_shared<MultiroomProvider>(device, ipcFactory) : nullptr);
    auto spectrogramAnimationProvider = (isStereoPairSupported ? std::make_shared<SpectrogramAnimationProvider>(ipcFactory) : nullptr);
    auto stereoPairProvider = std::make_shared<StereoPairProvider>(device, ipcFactory);
    auto updatesProvider = std::make_shared<UpdatesProvider>(ipcFactory);
    auto userConfigProvider = std::make_shared<UserConfigProvider>(ipcFactory);
    auto volumeManagerProvider = std::make_shared<VolumeManagerProvider>(ipcFactory);
    auto delayTimingsPolicy = std::make_shared<BackoffRetriesWithRandomPolicy>(getCrc32(device->deviceId()) + getNowTimestampMs());

    yandexIOEndpoint.addService<YandexIO::IOHubService>(ipcFactory);
    yandexIOEndpoint.addService<AdbService>(device, userConfigProvider);
    yandexIOEndpoint.addService<BrainService>(device, ipcFactory);
    yandexIOEndpoint.addService<PushService>(device, ipcFactory, authProvider, userConfigProvider);
    yandexIOEndpoint.addService<SyncService>(device, ipcFactory, authProvider, deviceStateProvider, delayTimingsPolicy);
    yandexIOEndpoint.addService<AlarmService>(device, ipcFactory, authProvider, stereoPairProvider);
    yandexIOEndpoint.addService<NtpService>(device, ipcFactory, userConfigProvider);
    yandexIOEndpoint.addService<NotificationService>(device, ipcFactory, authProvider, stereoPairProvider, userConfigProvider, sdk);
    yandexIOEndpoint.addService<DoNotDisturbService>(device, ipcFactory);
    if (isStereoPairSupported) {
        yandexIOEndpoint.addService<StereoPairService>(
            device,
            ipcFactory,
            clockTowerProvider,
            glagolClusterProvider,
            spectrogramAnimationProvider,
            userConfigProvider,
            volumeManagerProvider,
            sdk);
    }

    std::shared_ptr<SelfDestroyer> selfDestroyer = SelfDestroyerUtils::create(
        PROCESS_NAME,
        device,
        "aliced");

    if (AudioSender::configState(device) == AudioSender::ConfigState::DISABLED) {
        const auto& aliceConfig = device->configuration()->getServiceConfig("aliced");
        yandexIOEndpoint.addService<AliceService>(
            device,
            NAlice::TEndpoint::SpeakerEndpointType,
            sdk,
            ipcFactory,
            authProvider,
            clockTowerProvider,
            deviceStateProvider,
            glagolClusterProvider,
            multiroomProvider,
            stereoPairProvider,
            userConfigProvider,
            createAudioPlayer(aliceConfig),
            selfDestroyer);
    } else {
        yandexIOEndpoint.addService<AudioSenderService>(
            device, ipcFactory, ::SpeechKit::PlatformTypes::createDefaultAudioPlayer(), authProvider);
    }

#if !defined(__ANDROID__)
    yandexIOEndpoint.addService<BleService>(device, ipcFactory, deviceStateProvider);
#endif

    yandexIOEndpoint.addService<BrickService>(device, ipcFactory);
    yandexIOEndpoint.addService<SetupService>(device, ipcFactory, sdk);
    yandexIOEndpoint.addService<SoundInitService>(device, ipcFactory, deviceStateProvider, sdk);
    yandexIOEndpoint.addService<AuthService>(device, ipcFactory);
    yandexIOEndpoint.addService<FirstRunService>(device, ipcFactory, authProvider, deviceStateProvider, updatesProvider, userConfigProvider, sdk);
    yandexIOEndpoint.addService<IotService>(device, ipcFactory, sdk);
    yandexIOEndpoint.addService<NetworkService>(ipcFactory);
    yandexIOEndpoint.addService<BugReportService>(device, ipcFactory, userConfigProvider);

    yandexIOEndpoint.addService<UpdaterSwitcherFlagService>(userConfigProvider);

    return selfDestroyer;
}
