#include "yiod_factory.h"

#include <yandex_io/scaffolding/base/endpoint.h>
#include <yandex_io/scaffolding/base/utils.h>

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/interfaces/updates/connector/updates_provider.h>
#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/minidump/quasar_minidump.h>
#include <yandex_io/libs/self_destroyer/self_destroyer_utils.h>
#include <yandex_io/libs/speechkit_audio_player/devnull_audio_player.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/sdk/yandex_iosdk.h>
#include <yandex_io/sdk/hub/io_hub_service.h>
#include <yandex_io/services/aliced/alice_service.h>
#include <yandex_io/services/authd/auth_service.h>
#include <yandex_io/services/braind/brain_service.h>
#include <yandex_io/services/firstrund/first_run_service.h>
#include <yandex_io/services/setupd/setup_service.h>
#include <yandex_io/services/sound_initd/sound_init_service.h>
#include <yandex_io/services/syncd/sync_service.h>

#if !defined(__ANDROID__)
    #include <yandex_io/services/ble_initd/ble_service.h>
#endif

#include <memory>
#include <unistd.h>

using namespace quasar;

namespace {
    constexpr const char* ENDPOINT_NAME = "yiod";
    constexpr const char* PROCESS_NAME = "yiod_factory";
} // namespace

int yiod_factory(QuasarCallParams& params)
{
    TerminateWaiter waiter;

    const auto device = params.device();
    const auto ipcFactory = params.mixedIpcFactory();
    ipcFactory->setProcessName(PROCESS_NAME);

    const auto sdk = std::make_shared<YandexIO::YandexIOSDK>();
    sdk->init(ipcFactory, PROCESS_NAME, device->deviceId());

    sendDaemonStartMetric(PROCESS_NAME, device);
    auto yiodConfig = device->configuration()->getServiceConfig(ENDPOINT_NAME);
    Logging::initLogging(yiodConfig);
    Logging::addLoggingToTelemetryIfNeeded(yiodConfig, device->telemetry());

    YIO_LOG_INFO("Starting Factory daemon yiod version " << device->softwareVersion());

    YandexIOEndpoint yandexIOEndpoint(device, ENDPOINT_NAME);

    QuasarMinidump::getInstance().init(ENDPOINT_NAME, yiodConfig, device->telemetry(), device->softwareVersion());

    auto authProvider = std::make_shared<AuthProvider>(ipcFactory);
    auto deviceStateProvider = std::make_shared<DeviceStateProvider>(ipcFactory);
    auto updatesProvider = std::make_shared<UpdatesProvider>(ipcFactory);
    auto userConfigProvider = std::make_shared<UserConfigProvider>(ipcFactory);
    auto delayTimingsPolicy = std::make_shared<BackoffRetriesWithRandomPolicy>(getCrc32(device->deviceId()) + getNowTimestampMs());

    yandexIOEndpoint.addService<YandexIO::IOHubService>(ipcFactory);

    yandexIOEndpoint.addService<BrainService>(device, ipcFactory);
    yandexIOEndpoint.addService<SoundInitService>(device, ipcFactory, deviceStateProvider, sdk);
    yandexIOEndpoint.addService<AuthService>(device, ipcFactory);
    yandexIOEndpoint.addService<FirstRunService>(device, ipcFactory, authProvider, deviceStateProvider, updatesProvider, userConfigProvider, sdk);
    yandexIOEndpoint.addService<SetupService>(device, ipcFactory, sdk);
    yandexIOEndpoint.addService<SyncService>(device, ipcFactory, authProvider, deviceStateProvider, delayTimingsPolicy);

    std::shared_ptr<SelfDestroyer> selfDestroyer = SelfDestroyerUtils::create(
        PROCESS_NAME,
        device,
        "aliced");

    yandexIOEndpoint.addService<AliceService>(
        device,
        NAlice::TEndpoint::SpeakerEndpointType,
        sdk,
        ipcFactory,
        authProvider,
        nullptr,
        deviceStateProvider,
        nullptr,
        nullptr,
        nullptr,
        userConfigProvider,
        DevnullAudioPlayer::create(),
        selfDestroyer);

#if !defined(__ANDROID__)
    yandexIOEndpoint.addService<BleService>(device, ipcFactory, deviceStateProvider);
#endif

    yandexIOEndpoint.startAll();

    /// Block voice assistant as we use DevnullAudioPlayer
    /// if not blocked, ConfigurationModeHandler will try to get server onboarding instead of playing soundFile
    sdk->blockVoiceAssistant(PROCESS_NAME, std::nullopt);

    quasar::PeriodicExecutor killer([&]() {
        selfDestroyer->destroySelfIfNeed();
    }, std::chrono::minutes(30));

    waiter.wait();

    _exit(0);

    return 0;
}
