#include "audio_client_event_converter.h"

#include <stdexcept>

using namespace YandexIO;
using namespace quasar;

namespace {
    AudioMetadata convertAudioMetadata(const proto::AudioMetadata& audioMetadata) {
        AudioMetadata res;
        res.title = audioMetadata.title();
        res.subtitle = audioMetadata.subtitle();
        res.artImageUrl = audioMetadata.art_image_url();
        res.hideProgressBar = audioMetadata.hide_progress_bar();
        return res;
    }

    proto::AudioMetadata convertAudioMetadata(const AudioMetadata& audioMetadata) {
        proto::AudioMetadata res;
        res.set_title(TString(audioMetadata.title));
        res.set_subtitle(TString(audioMetadata.subtitle));
        res.set_art_image_url(TString(audioMetadata.artImageUrl));
        res.set_hide_progress_bar(audioMetadata.hideProgressBar);
        return res;
    }

    Audio::Screen convertAudioScreen(const proto::Audio::Screen& audioScreen) {
        switch (audioScreen) {
            case proto::Audio_Screen::Audio_Screen_MUSIC:
                return Audio::Screen::MUSIC;
            case proto::Audio_Screen::Audio_Screen_RADIO:
                return Audio::Screen::RADIO;
            default:
                return Audio::Screen::DEFAULT;
        }
    }

    proto::Audio::Screen convertAudioScreen(const Audio::Screen& audioScreen) {
        switch (audioScreen) {
            case Audio::Screen::MUSIC:
                return proto::Audio_Screen::Audio_Screen_MUSIC;
            case Audio::Screen::RADIO:
                return proto::Audio_Screen::Audio_Screen_RADIO;
            default:
                return proto::Audio_Screen::Audio_Screen_DEFAULT;
        }
    }

    Audio convertAudio(const proto::Audio& audio) {
        Audio res;
        res.positionSec = audio.position_sec();
        res.durationSec = audio.duration_sec();
        res.providerName = audio.provider_name();
        res.screenType = convertAudioScreen(audio.screen_type());
        res.metadata = convertAudioMetadata(audio.metadata());
        return res;
    }

    proto::Audio convertAudio(const Audio& audio) {
        proto::Audio res;
        res.set_position_sec(audio.positionSec);
        res.set_duration_sec(audio.durationSec);
        res.set_provider_name(TString(audio.providerName));
        res.set_screen_type(convertAudioScreen(audio.screenType));
        res.mutable_metadata()->CopyFrom(convertAudioMetadata(audio.metadata));
        return res;
    }

    AudioClientState convertAudioClientState(const proto::AudioClientState& audio) {
        switch (audio) {
            case proto::AudioClientState::BUFFERING:
                return AudioClientState::BUFFERING;
            case proto::AudioClientState::PLAYING:
                return AudioClientState::PLAYING;
            case proto::AudioClientState::PAUSED:
                return AudioClientState::PAUSED;
            case proto::AudioClientState::STOPPED:
                return AudioClientState::STOPPED;
            case proto::AudioClientState::FINISHED:
                return AudioClientState::FINISHED;
            case proto::AudioClientState::FAILED:
                return AudioClientState::FAILED;
            case proto::AudioClientState::IDLE:
                return AudioClientState::IDLE;
        }
        throw std::invalid_argument("Unknown proto::AudioClientState");
    }

    proto::AudioClientState convertAudioClientState(const AudioClientState& audio) {
        switch (audio) {
            case AudioClientState::BUFFERING:
                return proto::AudioClientState::BUFFERING;
            case AudioClientState::PLAYING:
                return proto::AudioClientState::PLAYING;
            case AudioClientState::PAUSED:
                return proto::AudioClientState::PAUSED;
            case AudioClientState::STOPPED:
                return proto::AudioClientState::STOPPED;
            case AudioClientState::FINISHED:
                return proto::AudioClientState::FINISHED;
            case AudioClientState::FAILED:
                return proto::AudioClientState::FAILED;
            case AudioClientState::IDLE:
                return proto::AudioClientState::IDLE;
        }
        throw std::invalid_argument("Unknown AudioClientState");
    }

    AudioPlayerDescriptor::PlayerType convertPlayerType(const proto::AudioPlayerDescriptor::PlayerType& playerType) {
        switch (playerType) {
            case proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_AUDIO:
                return AudioPlayerDescriptor::PlayerType::AUDIO;
            case proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_VOICE:
                return AudioPlayerDescriptor::PlayerType::VOICE;
            case proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_STREAM:
                return AudioPlayerDescriptor::PlayerType::STREAM;
            case proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_FILE_PLAYER:
                return AudioPlayerDescriptor::PlayerType::FILE_PLAYER;
        }
        throw std::invalid_argument("Unknown proto::AudioPlayerDescriptor::PlayerType");
    }

    proto::AudioPlayerDescriptor::PlayerType convertPlayerType(const AudioPlayerDescriptor::PlayerType& playerType) {
        switch (playerType) {
            case AudioPlayerDescriptor::PlayerType::AUDIO:
                return proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_AUDIO;
            case AudioPlayerDescriptor::PlayerType::VOICE:
                return proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_VOICE;
            case AudioPlayerDescriptor::PlayerType::STREAM:
                return proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_STREAM;
            case AudioPlayerDescriptor::PlayerType::FILE_PLAYER:
                return proto::AudioPlayerDescriptor_PlayerType::AudioPlayerDescriptor_PlayerType_FILE_PLAYER;
        }
        throw std::invalid_argument("Unknown AudioPlayerDescriptor::PlayerType");
    }

    AudioPlayerDescriptor convertAudioPlayerDescriptor(const proto::AudioPlayerDescriptor& playerDescriptor) {
        AudioPlayerDescriptor res;
        res.type = convertPlayerType(playerDescriptor.type());
        return res;
    }

    proto::AudioPlayerDescriptor convertAudioPlayerDescriptor(const AudioPlayerDescriptor& playerDescriptor) {
        proto::AudioPlayerDescriptor res;
        res.set_type(convertPlayerType(playerDescriptor.type));
        return res;
    }

    proto::AudioClientEvent::Event convertEvent(const AudioClientEvent::Event& event) {
        switch (event) {
            case AudioClientEvent::Event::STATE_CHANGED:
                return proto::AudioClientEvent_Event::AudioClientEvent_Event_STATE_CHANGED;
            case AudioClientEvent::Event::NEARLY_FINISHED:
                return proto::AudioClientEvent_Event::AudioClientEvent_Event_NEARLY_FINISHED;
            case AudioClientEvent::Event::HEARTBEAT:
                return proto::AudioClientEvent_Event::AudioClientEvent_Event_HEARTBEAT;
        }
        throw std::invalid_argument("Unknown AudioClientEvent::Event");
    }

    AudioClientEvent::Event convertEvent(const proto::AudioClientEvent::Event& event) {
        switch (event) {
            case proto::AudioClientEvent_Event::AudioClientEvent_Event_STATE_CHANGED:
                return AudioClientEvent::Event::STATE_CHANGED;
            case proto::AudioClientEvent_Event::AudioClientEvent_Event_NEARLY_FINISHED:
                return AudioClientEvent::Event::NEARLY_FINISHED;
            case proto::AudioClientEvent_Event::AudioClientEvent_Event_HEARTBEAT:
                return AudioClientEvent::Event::HEARTBEAT;
        }
        throw std::invalid_argument("Unknown proto::AudioClientEvent::Event");
    }

} /* namespace */

namespace YandexIO {
    AudioClientEvent convertAudioClientEventFromProto(const proto::AudioClientEvent& event) {
        AudioClientEvent res;
        res.event = convertEvent(event.event());
        res.state = convertAudioClientState(event.state());
        res.audio = convertAudio(event.audio());
        res.playerDescriptor = convertAudioPlayerDescriptor(event.player_descriptor());
        return res;
    }

    proto::AudioClientEvent convertAudioClientEventToProto(const AudioClientEvent& event) {
        proto::AudioClientEvent res;
        res.set_event(convertEvent(event.event));
        res.set_state(convertAudioClientState(event.state));
        res.mutable_audio()->CopyFrom(convertAudio(event.audio));
        res.mutable_player_descriptor()->CopyFrom(convertAudioPlayerDescriptor(event.playerDescriptor));
        return res;
    }

} /* namespace YandexIO */
