#include "device_group_state_converter.h"

#include <stdexcept>

using namespace YandexIO;
using namespace quasar;

namespace {

    DeviceGroupState::Role convertRole(const proto::DeviceGroupState::Role& role) {
        switch (role) {
            case proto::DeviceGroupState_Role::DeviceGroupState_Role_STAND_ALONE:
                return DeviceGroupState::Role::STAND_ALONE;
            case proto::DeviceGroupState_Role::DeviceGroupState_Role_LEADER:
                return DeviceGroupState::Role::LEADER;
            case proto::DeviceGroupState_Role::DeviceGroupState_Role_FOLLOWER:
                return DeviceGroupState::Role::FOLLOWER;
        }
        throw std::invalid_argument("Unknown proto::DeviceGroupState::Role");
    }

    proto::DeviceGroupState::Role convertRole(const DeviceGroupState::Role& role) {
        switch (role) {
            case DeviceGroupState::Role::STAND_ALONE:
                return proto::DeviceGroupState_Role::DeviceGroupState_Role_STAND_ALONE;
            case DeviceGroupState::Role::LEADER:
                return proto::DeviceGroupState_Role::DeviceGroupState_Role_LEADER;
            case DeviceGroupState::Role::FOLLOWER:
                return proto::DeviceGroupState_Role::DeviceGroupState_Role_FOLLOWER;
        }
        throw std::invalid_argument("Unknown DeviceGroupState::Role");
    }

    DeviceGroupState::ConnectionState convertConnectionState(const proto::DeviceGroupState::ConnectionState& connectionState) {
        switch (connectionState) {
            case proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_NONE:
                return DeviceGroupState::ConnectionState::NONE;
            case proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_CONNECTING:
                return DeviceGroupState::ConnectionState::CONNECTING;
            case proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_CONNECTED:
                return DeviceGroupState::ConnectionState::CONNECTED;
        }
        throw std::invalid_argument("Unknown proto::DeviceGroupState::ConnectionState");
    }

    proto::DeviceGroupState::ConnectionState convertConnectionState(const DeviceGroupState::ConnectionState& connectionState) {
        switch (connectionState) {
            case DeviceGroupState::ConnectionState::NONE:
                return proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_NONE;
            case DeviceGroupState::ConnectionState::CONNECTING:
                return proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_CONNECTING;
            case DeviceGroupState::ConnectionState::CONNECTED:
                return proto::DeviceGroupState_ConnectionState::DeviceGroupState_ConnectionState_CONNECTED;
        }
        throw std::invalid_argument("Unknown DeviceGroupState::ConnectionState");
    }

    DeviceGroupState::Leader convertLeader(const proto::DeviceGroupState::Leader& leader) {
        DeviceGroupState::Leader res;
        res.connectionState = convertConnectionState(leader.connection_state());
        res.platform = leader.platform();
        res.deviceId = leader.device_id();
        return res;
    }

    proto::DeviceGroupState::Leader convertLeader(const DeviceGroupState::Leader& leader) {
        proto::DeviceGroupState::Leader res;
        res.set_connection_state(convertConnectionState(leader.connectionState));
        res.set_platform(TString(leader.platform));
        res.set_device_id(TString(leader.deviceId));
        return res;
    }

    DeviceGroupState::Follower convertFollower(const proto::DeviceGroupState::Follower& follower) {
        DeviceGroupState::Follower res;
        res.connectionState = convertConnectionState(follower.connection_state());
        res.platform = follower.platform();
        res.deviceId = follower.device_id();
        return res;
    }

    proto::DeviceGroupState::Follower convertFollower(const DeviceGroupState::Follower& follower) {
        proto::DeviceGroupState::Follower res;
        res.set_connection_state(convertConnectionState(follower.connectionState));
        res.set_platform(TString(follower.platform));
        res.set_device_id(TString(follower.deviceId));
        return res;
    }

} /* namespace */

namespace YandexIO {
    DeviceGroupState convertDeviceGroupStateFromProto(const quasar::proto::DeviceGroupState& state) {
        DeviceGroupState res;
        res.groupId = state.group_id();
        res.localRole = convertRole(state.local_role());
        res.leader = convertLeader(state.leader());
        res.follower = convertFollower(state.follower());
        return res;
    }

    quasar::proto::DeviceGroupState convertDeviceGroupStateToProto(const DeviceGroupState& state) {
        proto::DeviceGroupState res;
        res.set_group_id(state.groupId);
        res.set_local_role(convertRole(state.localRole));
        res.mutable_leader()->CopyFrom(convertLeader(state.leader));
        res.mutable_follower()->CopyFrom(convertFollower(state.follower));
        return res;
    }

} /* namespace YandexIO */
