#include "directive.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <sstream>

using quasar::getJson;
using quasar::getString;
using quasar::tryGetBool;
using quasar::tryGetString;
using quasar::tryGetVector;

namespace YandexIO {

    Directive::Data::Data(const std::string& name, const std::string& type)
        : name(name)
        , analyticsContextName(name)
        , type(type)
    {
    }

    Directive::Data::Data(const std::string& name, const std::string& type, Json::Value payload)
        : name(name)
        , analyticsContextName(name)
        , type(type)
        , payload(std::move(payload))
    {
    }

    Directive::Data::Data(const std::string& name, const std::string& type, std::optional<quasar::proto::AudioChannel> channel)
        : name(name)
        , analyticsContextName(name)
        , type(type)
        , channel(channel)
    {
    }

    bool Directive::Data::isServerAction() const {
        return type == "server_action";
    }

    void Directive::Data::setContext(
        std::string asrText_,
        std::string requestId_,
        std::string parentRequestId_,
        std::string displayedText_)
    {
        asrText = std::move(asrText_);
        requestId = std::move(requestId_);
        parentRequestId = std::move(parentRequestId_);
        displayedText = std::move(displayedText_);
    }

    Directive::Data Directive::Data::fromJson(const Json::Value& json)
    {
        Data data;

        data.endpointId = tryGetString(json, "endpoint_id", "");
        data.name = tryGetString(json, "name", "");
        data.analyticsContextName = tryGetString(json, "sub_name", "");
        data.type = tryGetString(json, "type", "");
        data.text = tryGetString(json, "text", "");
        data.multiroomSessionId = tryGetString(json, "multiroom_session_id", "");
        data.roomDeviceIds = tryGetVector<std::string>(json, "room_device_ids");
        data.ignoreAnswer = tryGetBool(json, "ignore_answer", false);
        data.isLedSilent = tryGetBool(json, "is_led_silent", false);
        data.isParallel = tryGetBool(json, "is_parallel", false);

        if (!json["payload"].isNull()) {
            try {
                data.payload = getJson(json, "payload");
                if (json.isMember("room_device_ids")) {
                    data.payload["room_device_ids"] = json["room_device_ids"];
                }
            } catch (const std::exception& e) {
                data.payload["state"] = getString(json, "payload");
            }
        }

        return data;
    }

    Directive::Directive(Directive::Data data, bool blocksSubsequentPrefetch)
        : data_(std::move(data))
        , blocksSubsequentPrefetch_(blocksSubsequentPrefetch)
    {
    }

    const Directive::Data& Directive::getData() const {
        return data_;
    }

    bool Directive::is(const std::string& name) const {
        return getData().name == name;
    }

    std::string Directive::format() const {
        std::stringstream ss;
        ss << "(" << this << ")"
           << ":" << getData().name;

        return ss.str();
    }

    void Directive::setName(std::string name) {
        data_.name = std::move(name);
    }

    void Directive::setPayload(Json::Value payload)
    {
        data_.payload = std::move(payload);
    }

    bool Directive::getPrefetchInProgress() const {
        return prefetchInProgress_;
    }

    void Directive::setPrefetchInProgress(bool value)
    {
        prefetchInProgress_ = value;
    }

    bool Directive::getFirstInChain() const {
        return firstInChain_;
    }

    void Directive::setFirstInChain()
    {
        firstInChain_ = true;
    }

    void Directive::setIsGetNext()
    {
        isGetNext_ = true;
    }

    bool Directive::isGetNext() const {
        return isGetNext_;
    }

    bool Directive::isRequest() const {
        return false;
    }

    void Directive::clearPrefetchResult()
    {
        prefetchResult_.clear();
    }

    void Directive::setPrefetchResult(std::list<std::shared_ptr<Directive>> value)
    {
        prefetchResult_ = std::move(value);
    }

    const std::list<std::shared_ptr<Directive>>& Directive::getPrefetchResult() const {
        return prefetchResult_;
    }

    const std::string& Directive::getRequestId() const {
        return data_.requestId;
    }

    const std::string& Directive::getParentRequestId() const {
        return data_.parentRequestId;
    }

    quasar::proto::Directive Directive::toProtobuf() const {
        return convertDirectiveToProtobuf(getData());
    }

    bool Directive::isBlocksSubsequentPrefetch() const {
        return blocksSubsequentPrefetch_;
    }

    void Directive::setBlocksSubsequentPrefetch(bool value)
    {
        blocksSubsequentPrefetch_ = value;
    }

    void Directive::setIsActive(bool value)
    {
        isActive_ = value;
    }

    bool Directive::isActive() const {
        return isActive_;
    }

    std::shared_ptr<Directive> Directive::createServerAction(const std::string& name)
    {
        Data data(name, "server_action");

        return std::make_shared<Directive>(std::move(data));
    }

    std::shared_ptr<Directive> Directive::createLocalAction(const std::string& name, const Json::Value& payload, const std::string& endpointId)
    {
        Data data(name, "local_action");
        data.payload = payload;
        data.endpointId = endpointId;

        return std::make_shared<Directive>(std::move(data));
    }

    std::shared_ptr<Directive> Directive::createClientAction(const std::string& name, const Json::Value& payload)
    {
        Data data(name, "client_action");
        data.payload = payload;

        return std::make_shared<Directive>(std::move(data));
    }

    std::string Directive::formatDirectives(const std::list<std::shared_ptr<Directive>>& directives)
    {
        bool first = true;
        std::stringstream ss;
        for (const auto& directive : directives) {
            ss << (!first ? ";" : "") << directive->format();
            first = false;
        }

        return std::move(ss).str();
    }

    std::string Directive::formatDirectiveNames(const std::list<std::shared_ptr<Directive>>& directives)
    {
        bool first = true;
        std::stringstream ss;
        ss << "[";
        for (const auto& directive : directives) {
            ss << (!first ? ", " : "") << directive->getData().name << ":" << directive->getFirstInChain();
            first = false;
        }
        ss << "]";

        return std::move(ss).str();
    }

    quasar::proto::Directive Directive::convertToDirectiveProtobuf(const std::shared_ptr<Directive>& directive)
    {
        return convertDirectiveToProtobuf(directive->getData());
    }

    quasar::proto::Directive Directive::convertDirectiveToProtobuf(const Directive::Data& data)
    {
        quasar::proto::Directive protobuf;

        protobuf.set_endpoint_id(TString(data.endpointId));
        protobuf.set_name(TString(data.name));
        protobuf.set_analytics_context_name(TString(data.analyticsContextName));
        protobuf.set_type(TString(data.type));
        protobuf.set_request_id(TString(data.requestId));
        protobuf.set_parent_request_id(TString(data.parentRequestId));
        protobuf.set_parent_message_id(TString(data.parentMessageId));
        protobuf.set_json_payload(quasar::jsonToString(data.payload));
        protobuf.set_text(TString(data.text));
        protobuf.set_asr_text(TString(data.asrText));
        protobuf.set_displayed_text(TString(data.displayedText));
        protobuf.set_ignore_answer(data.ignoreAnswer);
        protobuf.set_is_led_silent(data.isLedSilent);
        protobuf.set_is_route_locally(data.isRouteLocally);
        protobuf.set_is_prefetched(data.isPrefetched);
        if (data.channel.has_value()) {
            protobuf.set_channel(data.channel.value());
        }

        protobuf.set_multiroom_session_id(TString(data.multiroomSessionId));
        for (const auto& roomDeviceId : data.roomDeviceIds) {
            protobuf.add_room_device_ids(TString(roomDeviceId));
        }

        return protobuf;
    }

    quasar::proto::ExternalCommandMessage Directive::convertToExternalCommandProtobuf(const std::shared_ptr<Directive>& directive)
    {
        const auto& data = directive->getData();

        quasar::proto::ExternalCommandMessage command;
        command.set_name(TString(data.name));
        command.set_last_recognized_phrase(TString(data.asrText));
        command.set_vins_request_id(TString(data.requestId));
        command.set_displayed_text(TString(data.displayedText));
        command.set_payload(quasar::jsonToString(data.payload));
        command.set_multiroom_session_id(TString(data.multiroomSessionId));
        command.set_is_route_locally(data.isRouteLocally);
        for (const auto& roomDeviceId : data.roomDeviceIds) {
            command.add_room_device_ids(TString(roomDeviceId));
        }

        return command;
    }

    std::shared_ptr<Directive> Directive::createDirectiveFromProtobuf(const quasar::proto::Directive& protobuf) {
        Directive::Data data(protobuf.name(), protobuf.type());

        data.endpointId = protobuf.endpoint_id();
        data.analyticsContextName = protobuf.analytics_context_name();
        data.requestId = protobuf.request_id();
        data.parentRequestId = protobuf.parent_request_id();
        data.parentMessageId = protobuf.parent_message_id();
        data.text = protobuf.text();
        data.asrText = protobuf.asr_text();
        data.displayedText = protobuf.displayed_text();
        data.ignoreAnswer = protobuf.ignore_answer();
        data.isLedSilent = protobuf.is_led_silent();
        data.isRouteLocally = protobuf.is_route_locally();
        data.isPrefetched = protobuf.is_prefetched();
        data.multiroomSessionId = protobuf.multiroom_session_id();
        if (protobuf.has_channel()) {
            data.channel = protobuf.channel();
        }
        data.payload = quasar::tryParseJson(protobuf.json_payload()).value_or(Json::Value{});

        data.roomDeviceIds.reserve(protobuf.room_device_ids_size());
        for (const auto& roomDeviceId : protobuf.room_device_ids()) {
            data.roomDeviceIds.push_back(roomDeviceId);
        }

        return std::make_shared<Directive>(data);
    }

    std::shared_ptr<Directive> Directive::createFromExternalCommandMessage(const quasar::proto::ExternalCommandMessage& command)
    {
        Directive::Data data(command.name(), "local_action");
        if (command.has_payload()) {
            const auto optionalPayload = quasar::tryParseJson(command.payload());
            if (optionalPayload.has_value()) {
                data.payload = optionalPayload.value();
            }
        }
        if (command.has_vins_request_id()) {
            data.requestId = command.vins_request_id();
        }
        if (command.has_last_recognized_phrase()) {
            data.asrText = command.last_recognized_phrase();
        }
        if (command.has_displayed_text()) {
            data.displayedText = command.displayed_text();
        }
        if (command.has_multiroom_session_id()) {
            data.multiroomSessionId = command.multiroom_session_id();
        }
        if (command.has_is_route_locally()) {
            data.isRouteLocally = command.is_route_locally();
        }
        for (const auto& roomDeviceId : command.room_device_ids()) {
            data.roomDeviceIds.push_back(roomDeviceId);
        }

        return std::make_shared<YandexIO::Directive>(std::move(data));
    }

    std::string Directive::convertToJsonString(const std::shared_ptr<Directive>& directive) {
        return quasar::convertMessageToJsonString(convertToDirectiveProtobuf(directive));
    }

} // namespace YandexIO
