#pragma once

#include <yandex_io/protos/quasar_proto.pb.h>

#include <json/json.h>

#include <list>
#include <memory>
#include <string>

namespace YandexIO {

    class Directive {
    public:
        struct Data {
        public:
            Data() = default;
            Data(const std::string& name, const std::string& type);
            Data(const std::string& name, const std::string& type, Json::Value payload);
            Data(const std::string& name, const std::string& type, std::optional<quasar::proto::AudioChannel> channel);

            bool isServerAction() const;
            void setContext(std::string asrText,
                            std::string requestId,
                            std::string parentRequestId,
                            std::string displayedText);
            static Data fromJson(const Json::Value& json);

        public:
            std::string endpointId;
            std::string name;
            std::string analyticsContextName;
            std::string type;
            std::string text;
            std::string multiroomSessionId;
            std::vector<std::string> roomDeviceIds;
            bool ignoreAnswer = false;
            bool isLedSilent = true;
            Json::Value payload;
            std::string asrText;
            std::string requestId;
            std::string parentRequestId;
            std::string parentMessageId;
            std::string displayedText;
            std::optional<quasar::proto::AudioChannel> channel;
            bool isRouteLocally = false;
            bool isParallel = false;
            bool isPrefetched = false;
        };

    public:
        Directive() = default;
        Directive(Data data, bool blocksSubsequentPrefetch = false);
        virtual ~Directive() = default;

        virtual std::string format() const;
        virtual const Data& getData() const;
        bool is(const std::string& name) const;

        virtual bool isRequest() const;
        virtual bool getPrefetchInProgress() const;
        virtual void setPrefetchInProgress(bool value);

        virtual bool getFirstInChain() const;
        virtual void setFirstInChain();

        void setIsGetNext();
        void setName(std::string name);
        void setPayload(Json::Value payload);
        bool isGetNext() const;

        bool isBlocksSubsequentPrefetch() const;
        void setBlocksSubsequentPrefetch(bool value);

        void setIsActive(bool value);
        bool isActive() const;

        virtual void clearPrefetchResult();
        virtual void setPrefetchResult(std::list<std::shared_ptr<Directive>> value);
        const std::list<std::shared_ptr<Directive>>& getPrefetchResult() const;

        const std::string& getRequestId() const;
        const std::string& getParentRequestId() const;

        virtual quasar::proto::Directive toProtobuf() const;

    public:
        static std::shared_ptr<Directive> createServerAction(const std::string& name);
        static std::shared_ptr<Directive> createClientAction(
            const std::string& name, const Json::Value& payload = Json::Value());
        static std::shared_ptr<Directive> createLocalAction(
            const std::string& name, const Json::Value& payload = Json::Value(), const std::string& endpointId = "");

        static std::string formatDirectives(const std::list<std::shared_ptr<Directive>>& directives);
        static std::string formatDirectiveNames(const std::list<std::shared_ptr<Directive>>& directives);

        static quasar::proto::Directive convertDirectiveToProtobuf(const Directive::Data& data);
        static quasar::proto::Directive convertToDirectiveProtobuf(const std::shared_ptr<Directive>& directive);
        static quasar::proto::ExternalCommandMessage convertToExternalCommandProtobuf(const std::shared_ptr<Directive>& directive);
        static std::shared_ptr<Directive> createDirectiveFromProtobuf(const quasar::proto::Directive& protobuf);
        static std::shared_ptr<Directive> createFromExternalCommandMessage(const quasar::proto::ExternalCommandMessage& command);
        static std::string convertToJsonString(const std::shared_ptr<Directive>& directive);

    protected:
        Data data_;

    private:
        bool prefetchInProgress_ = false;
        bool firstInChain_ = false;
        std::list<std::shared_ptr<Directive>> prefetchResult_;
        bool isGetNext_ = false;
        bool blocksSubsequentPrefetch_ = false;
        bool isActive_ = true;
    };

} // namespace YandexIO
