#include "mock_i_capability.h"

namespace YandexIO {

    MockICapability::MockICapability(std::string id, std::shared_ptr<IDirectiveHandler> directiveHandler)
        : id_(std::move(id))
        , directiveHandler_(std::move(directiveHandler))
    {
        ON_CALL(*this, getState())
            .WillByDefault(testing::Return(NAlice::TCapabilityHolder{}));
        ON_CALL(*this, getId())
            .WillByDefault(testing::ReturnRef(id_));
        ON_CALL(*this, getDirectiveHandler())
            .WillByDefault(testing::Return(directiveHandler_));

        ON_CALL(*this, addListener(testing::_)).WillByDefault(testing::Invoke([this](std::weak_ptr<ICapability::IListener> listener) {
            addListenerDefault(listener);
        }));
        ON_CALL(*this, removeListener(testing::_)).WillByDefault(testing::Invoke([this](std::weak_ptr<ICapability::IListener> listener) {
            removeListenerDefault(listener);
        }));
    }

    void MockICapability::addListenerDefault(std::weak_ptr<IListener> wlistener)
    {
        listeners_.push_back(wlistener);
    }

    void MockICapability::removeListenerDefault(std::weak_ptr<IListener> wlistener)
    {
        const auto iter = std::find_if(listeners_.begin(), listeners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });
        if (iter != listeners_.end()) {
            listeners_.erase(iter);
        }
    }

    void MockICapability::notifyStateChanged(const NAlice::TCapabilityHolder& state) {
        for (const auto& wlistener : listeners_) {
            if (const auto listener = wlistener.lock()) {
                listener->onCapabilityStateChanged(shared_from_this(), state);
            }
        }
    }

    void MockICapability::notifyEvents(const std::vector<NAlice::TCapabilityEvent>& events) {
        for (const auto& wlistener : listeners_) {
            if (const auto listener = wlistener.lock()) {
                listener->onCapabilityEvents(shared_from_this(), events);
            }
        }
    }

} // namespace YandexIO
