//
// © YANDEX LLC, 2018
//

#include "device_context.h"

#include <yandex_io/sdk/converters/glagol_discovery_result_converter.h>

#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;
using namespace quasar;

DeviceContext::DeviceContext(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, onConnectedCb onConnected, bool autoConnect)
    : connector_(ipcFactory->createIpcConnector("iohub_services"))
    , onConnected_(std::move(onConnected))
{
    if (!ipcFactory) {
        YIO_LOG_WARN("Possibly creating a Device Context with ipcFactory equal to nullptr is an error");
    }

    connector_->setConnectHandler(std::bind(&DeviceContext::handleConnect, this));
    connector_->setMessageHandler(std::bind(&DeviceContext::handleQuasarMessage, this, std::placeholders::_1));
    if (autoConnect) {
        connectToSDK();
    }
}

void DeviceContext::shutdown() {
    connector_->shutdown();
    connector_->waitUntilDisconnected();
}

void DeviceContext::setSetupMode(bool isSetup, bool isFirstSetup) {
    if (isSetup) {
        fireStartSetup(isFirstSetup);
    } else {
        fireFinishSetup();
    }
}

void DeviceContext::fireSDKState(const proto::IOEvent::SDKState& state) {
    proto::QuasarMessage msg;
    msg.mutable_io_event()->mutable_sdk_state()->CopyFrom(state);
    connector_->sendMessage(std::move(msg));
}

void DeviceContext::fireSoundDataTransferStart() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_sound_data_transfer_start();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireConnectingToNetwork() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_connecting_to_network();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireSetupError() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_setup_error();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireStartSetup(bool isFirstSetup) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_start_setup(isFirstSetup);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireFinishSetup() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_finish_setup();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireConversationError() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_conversation_error();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireEnqueueAlarm(const quasar::proto::IOEvent_AlarmType& alarmType, const std::string& alarmId) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_alarm_enqueued()->set_type(alarmType);
    message.mutable_io_event()->mutable_on_alarm_enqueued()->set_id(TString(alarmId));
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireStartAlarm(const quasar::proto::IOEvent_AlarmType& alarmType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_alarm_started(alarmType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireStopAlarm(const quasar::proto::IOEvent_AlarmType& alarmType, bool hasRemainingMedia) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_alarm_stopped()->set_type(alarmType);
    message.mutable_io_event()->mutable_on_alarm_stopped()->set_has_remaining_media(hasRemainingMedia);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAlarmStopRemainingMedia() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_alarm_stop_remaining_media();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAlarmsSettingsChanged(const quasar::proto::AlarmsSettings& alarmsSettings) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_alarms_settings_changed()->CopyFrom(alarmsSettings);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaRequest(proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_request(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaStarted(proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_started(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaPaused(proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_paused(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaResumed(proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_resumed(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaError(proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_error(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaSwitchedForward(quasar::proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_switched_forward(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaSwitchedBackward(quasar::proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_switched_backward(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaLiked(quasar::proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_liked(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMediaDisliked(quasar::proto::MediaContentType contentType) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_media_disliked(contentType);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireLastUpdateState(bool wasAttempted, bool attemptSucceeded) {
    proto::QuasarMessage message;
    auto lastUpdateInfo = message.mutable_io_event()->mutable_last_update_info();
    lastUpdateInfo->set_update_was_attempted(wasAttempted);
    lastUpdateInfo->set_update_succeeded(attemptSucceeded);

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireExternalCommand(const std::string& name, const std::string& vinsRequestId, const std::string& payload) {
    proto::QuasarMessage message;
    auto externalCommand = message.mutable_io_event()->mutable_external_command();
    externalCommand->set_name(TString(name));
    externalCommand->set_payload(TString(payload));
    externalCommand->set_vins_request_id(TString(vinsRequestId));

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireSystemConfig(const std::string& configName, const std::string& jsonConfigValue) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_backend_config()->set_name(TString(configName));
    message.mutable_io_event()->mutable_backend_config()->set_value(TString(jsonConfigValue));

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireDeviceConfig(const std::string& configName, const std::string& jsonConfigValue) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_device_config()->set_name(TString(configName));
    message.mutable_io_event()->mutable_device_config()->set_value(TString(jsonConfigValue));

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAccountConfig(const std::string& configName, const std::string& jsonConfigValue) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_account_config()->set_name(TString(configName));
    message.mutable_io_event()->mutable_account_config()->set_value(TString(jsonConfigValue));

    connector_->sendMessage(std::move(message));
}

void DeviceContext::firePushNotification(const std::string& operation, const std::string& messageStr) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_push_notification()->set_operation(TString(operation));
    message.mutable_io_event()->mutable_push_notification()->set_message(TString(messageStr));

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireNotificationPending() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_notification_pending();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireNotificationStarted() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_notification_started();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireNotificationEnd() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_notification_end();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtAvrcpPause() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_sink_request()->mutable_pause();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtAvrcpResume() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_sink_request()->mutable_play();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtAvrcpNext() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_sink_request()->mutable_next();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtAvrcpPrev(bool forced) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_sink_request()->mutable_prev()->set_forced(forced);

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtTakeAudioFocus() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_take_audio_focus();

    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBtFreeAudioFocus() {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_bluetooth_request()->mutable_free_audio_focus();
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAuthenticationStatus(const std::string& oauthCode, bool is_ok, const std::string& response) {
    proto::QuasarMessage message;
    auto authStatus = message.mutable_io_event()->mutable_authentication_status();
    authStatus->set_oauth_code(TString(oauthCode));
    authStatus->set_is_ok(is_ok);
    authStatus->set_response(TString(response));
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireOAuthTokenIsInvalid(const std::string& oauthToken) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_invalid_oauth_token(TString(oauthToken));
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAuthenticationIsInvalid(const std::string& uid) {
    proto::QuasarMessage message;
    message.mutable_io_event()->set_on_invalid_authentication(TString(uid));
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireSpectrum(proto::IOEvent::Spectrum&& spectrum) {
    proto::QuasarMessage message;
    *message.mutable_io_event()->mutable_spectrum() = spectrum;
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireSoundCommand(const std::string& command) {
    proto::QuasarMessage message;
    *message.mutable_io_event()->mutable_sound_command() = command;
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireStartingConfigureSuccess(bool shouldUpdate) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_starting_configure_success()->set_should_update(shouldUpdate);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireMusicStateChanged(const quasar::proto::AppState::MusicState& musicState) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_music_state_changed()->mutable_music_state()->CopyFrom(musicState);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireAudioClientEvent(const quasar::proto::AudioClientEvent& audioClientEvent) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_on_audio_client_event()->CopyFrom(audioClientEvent);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireBrickStatusChanged(const quasar::proto::BrickInfo& brickInfo) {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_brick_info()->CopyFrom(brickInfo);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireDeviceGroupStateChanged(const quasar::proto::DeviceGroupState& groupState) const {
    proto::QuasarMessage message;
    message.mutable_io_event()->mutable_device_group_state()->CopyFrom(groupState);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireDiscoveryResult(const glagol::IDiscovery::Result& discoveryResult) const {
    proto::QuasarMessage message;
    const auto protoResult = convertDiscoveryResultToProto(discoveryResult);
    message.mutable_io_event()->mutable_glagol_discovery_result()->CopyFrom(protoResult);
    connector_->sendMessage(std::move(message));
}

void DeviceContext::fireGetSyncInfoStatus(bool isHttpRequestOk) {
    connector_->sendMessage(ipc::buildMessage([isHttpRequestOk](auto& msg) {
        constexpr auto STATUS_OK = proto::IOEvent::GetSyncInfoRequestStatus::OK;
        constexpr auto STATUS_ERROR = proto::IOEvent::GetSyncInfoRequestStatus::NETWORK_ERROR;
        const auto status = isHttpRequestOk ? STATUS_OK : STATUS_ERROR;
        msg.mutable_io_event()->mutable_get_sync_info_status()->set_status(status);
    }));
}

void DeviceContext::handleConnect() {
    /* Call DeviceContext user callback */
    if (onConnected_) {
        onConnected_();
    }
}

void DeviceContext::handleQuasarMessage(const ipc::SharedMessage& message) {
    if (message->has_io_hub_message() && message->io_hub_message().has_retrieve_state()) {
        // Hub requested to return saved state
        handleConnect();
    }

    if (!message->has_io_control()) {
        return;
    }

    const auto& ioControlMessage = message->io_control();

    if (ioControlMessage.has_allow_init_configuration_state()) {
        /* Guarantee that onQuasarStart will be called once */
        if (!onAllowInitConfigurationStateDone_) {
            onAllowInitConfigurationStateDone_ = true;
            if (onAllowInitConfigurationState) {
                onAllowInitConfigurationState();
            }
        }
    }
    if (ioControlMessage.has_set_setup_mode()) {
        /* should start setup mode */
        if (onSetupMode) {
            onSetupMode(true);
        }
    }
    if (ioControlMessage.has_stop_setup_mode()) {
        /* should stop setup mode */
        if (onSetupMode) {
            onSetupMode(false);
        }
    }
    if (ioControlMessage.has_toggle_setup_mode()) {
        if (onToggleSetupMode) {
            onToggleSetupMode();
        }
    }
    if (ioControlMessage.has_approve_alarm()) {
        if (onApproveAlarm) {
            onApproveAlarm(ioControlMessage.approve_alarm());
        }
    }
    if (ioControlMessage.has_subscribe_to_device_config()) {
        if (onSubscribeToDeviceConfig) {
            onSubscribeToDeviceConfig(ioControlMessage.subscribe_to_device_config());
        }
    }
    if (ioControlMessage.has_unsubscribe_from_device_config()) {
        if (onUnsubscribeFromDeviceConfig) {
            onUnsubscribeFromDeviceConfig(ioControlMessage.unsubscribe_from_device_config());
        }
    }
    if (ioControlMessage.has_subscribe_to_system_config()) {
        if (onSubscribeToSystemConfig) {
            onSubscribeToSystemConfig(ioControlMessage.subscribe_to_system_config());
        }
    }
    if (ioControlMessage.has_unsubscribe_from_system_config()) {
        if (onUnsubscribeFromSystemConfig) {
            onUnsubscribeFromSystemConfig(ioControlMessage.unsubscribe_from_system_config());
        }
    }
    if (ioControlMessage.has_subscribe_to_account_config()) {
        if (onSubscribeToAccountConfig) {
            onSubscribeToAccountConfig(ioControlMessage.subscribe_to_account_config());
        }
    }
    if (ioControlMessage.has_unsubscribe_from_account_config()) {
        if (onUnsubscribeFromAccountConfig) {
            onUnsubscribeFromAccountConfig(ioControlMessage.unsubscribe_from_account_config());
        }
    }
    if (ioControlMessage.has_authenticate()) {
        if (onChangeAccount) {
            onChangeAccount(ioControlMessage.authenticate());
        }
    }
    if (ioControlMessage.has_allow_update()) {
        if (onAllowUpdate) {
            onAllowUpdate(ioControlMessage.allow_update().for_all(), ioControlMessage.allow_update().for_critical());
        }
    }
    if (ioControlMessage.has_prepared_for_notification()) {
        if (onPreparedForNotification) {
            onPreparedForNotification();
        }
    }
    if (ioControlMessage.has_bluetooth_sink_event()) {
        if (onBluetoothSinkMediaEvent) {
            onBluetoothSinkMediaEvent(ioControlMessage.bluetooth_sink_event());
        }
    }
    if (ioControlMessage.has_numeric_metrics()) {
        if (onNumericMetrics) {
            onNumericMetrics(ioControlMessage.numeric_metrics().key(), ioControlMessage.numeric_metrics().value());
        }
    }
    if (ioControlMessage.has_categorical_metrics()) {
        if (onCategoricalMetrics) {
            onCategoricalMetrics(ioControlMessage.categorical_metrics().key(), ioControlMessage.categorical_metrics().value());
        }
    }
    if (ioControlMessage.has_provide_user_account_info()) {
        if (onUserAccountInfo) {
            const auto& info = ioControlMessage.provide_user_account_info();
            onUserAccountInfo(info.oauth_token(), info.uid());
        }
    }

    if (ioControlMessage.has_revoke_user_account_info()) {
        if (onUserAccountInfo) {
            onUserAccountInfo({}, {});
        }
    }

    if (ioControlMessage.has_tv_policy_info()) {
        if (onTvPolicyInfoReceived) {
            const auto& tvPolicyInfoMsg = ioControlMessage.tv_policy_info();
            TvPolicyInfo tvPolicyInfo;

            if (tvPolicyInfoMsg.has_content_settings()) {
                tvPolicyInfo.contentSettings = tvPolicyInfoMsg.content_settings();
            }

            if (tvPolicyInfoMsg.has_age_limit()) {
                tvPolicyInfo.ageLimit = tvPolicyInfoMsg.age_limit();
            }
            onTvPolicyInfoReceived(tvPolicyInfo);
        }
    }

    if (ioControlMessage.has_clock_display_state()) {
        if (onClockDisplayStateReceived) {
            const auto& state = ioControlMessage.clock_display_state();
            ClockDisplayState clockDisplayState;

            if (state.has_is_clock_on()) {
                clockDisplayState.isClockOn = state.is_clock_on();
            }

            if (state.has_clock_enabled()) {
                clockDisplayState.clockEnabled = state.clock_enabled();
            }

            if (state.has_show_clock_during_dnd()) {
                clockDisplayState.showClockDuringDnd = state.show_clock_during_dnd();
            }

            if (state.has_brightness()) {
                clockDisplayState.brightness = state.brightness();
            }

            onClockDisplayStateReceived(clockDisplayState);
        }
    }

    if (ioControlMessage.has_assistant_blocking()) {
        if (onAssistantBlocking) {
            onAssistantBlocking(ioControlMessage.assistant_blocking().source(),
                                ioControlMessage.assistant_blocking().has_error_sound() ? std::make_optional(ioControlMessage.assistant_blocking().error_sound()) : std::nullopt);
        }
    }

    if (ioControlMessage.has_assistant_unblocking()) {
        if (onAssistantUnblocking) {
            onAssistantUnblocking(ioControlMessage.assistant_unblocking().source());
        }
    }

    if (ioControlMessage.has_register_device_in_backend()) {
        if (onRegistrationInBackendRequested) {
            const auto& msg = ioControlMessage.register_device_in_backend();
            onRegistrationInBackendRequested(msg.oauth_token(), msg.uid(),
                                             [this, request_id{message->request_id()}](RegistrationResult result) {
                                                 proto::QuasarMessage responseMessage;
                                                 responseMessage.set_request_id(request_id);

                                                 auto registrationResult = responseMessage.mutable_io_event()->mutable_registration_result();
                                                 registrationResult->set_is_success(result.isSuccess);
                                                 registrationResult->set_response_code(result.responseCode);
                                                 registrationResult->set_error_message(TString(result.errorMessage));

                                                 connector_->sendMessage(std::move(responseMessage));
                                             });
        }
    }

    if (ioControlMessage.has_call_message()) {
        const auto& callMessage = ioControlMessage.call_message();
        if (callMessage.has_accept_incoming() && onAcceptIncomingCall) {
            onAcceptIncomingCall();
        } else if (callMessage.has_decline_incoming() && onDeclineIncomingCall) {
            onDeclineIncomingCall();
        } else if (callMessage.has_decline_current() && onDeclineCurrentCall) {
            onDeclineCurrentCall();
        }
    }

    if (ioControlMessage.has_is_screen_active()) {
        if (onIsScreenActive) {
            onIsScreenActive(ioControlMessage.is_screen_active());
        }
    }

    if (ioControlMessage.has_provide_device_state()) {
        if (onDeviceStatePart) {
            onDeviceStatePart(ioControlMessage.provide_device_state());
        }
    }

    if (ioControlMessage.has_timezone()) {
        if (onTimezone) {
            onTimezone(ioControlMessage.timezone());
        }
    }

    if (ioControlMessage.has_location()) {
        if (onLocation) {
            onLocation(ioControlMessage.location());
        }
    }

    if (ioControlMessage.has_wifi_list()) {
        if (onWifiList) {
            onWifiList(ioControlMessage.wifi_list());
        }
    }

    if (ioControlMessage.has_vins_response()) {
        if (onVinsResponse) {
            onVinsResponse(ioControlMessage.vins_response());
        }
    }

    if (ioControlMessage.has_active_actions()) {
        if (onActiveActionReceived) {
            onActiveActionReceived(ioControlMessage.active_actions());
        }
    }

    if (ioControlMessage.has_active_action()) {
        if (onActiveActionSemanticFrameReceived) {
            const auto& activeActionMsg = ioControlMessage.active_action();
            std::optional<std::string> activeActionPayload;

            if (activeActionMsg.has_semantic_frame_payload()) {
                activeActionPayload = activeActionMsg.semantic_frame_payload();
            }

            onActiveActionSemanticFrameReceived(activeActionPayload);
        }
    }

    if (ioControlMessage.has_equalizer_config()) {
        if (onEqualizerConfig) {
            onEqualizerConfig(ioControlMessage.equalizer_config());
        }
    }

    if (ioControlMessage.has_media_device_identifier()) {
        if (onMediaDeviceIdentifier) {
            onMediaDeviceIdentifier(ioControlMessage.media_device_identifier());
        }
    }
}

void DeviceContext::connectToSDK() {
    connector_->connectToService();
}

void DeviceContext::waitUntilConnected() const {
    connector_->waitUntilConnected();
}

std::string DeviceContext::preprocessVinsResponse(const std::string& response) {
    proto::QuasarMessage command;
    command.mutable_io_event()->set_vins_response_preprocess_request(TString(response));

    auto result = connector_->sendRequestSync(std::move(command), std::chrono::seconds(5));
    if (result && result->has_io_control() && result->io_control().has_vins_response_preprocess_response()) {
        return result->io_control().vins_response_preprocess_response();
    } else {
        return response;
    }
}
