#pragma once

//
// © YANDEX LLC, 2018
//

#pragma once

#include <yandex_io/sdk/clock_display_state.h>
#include <yandex_io/sdk/registration_result.h>
#include <yandex_io/sdk/tv_set_state.h>
#include <yandex_io/sdk/proto/device_sdk.pb.h>

#include <yandex_io/libs/bluetooth/bluetooth.h>
#include <yandex_io/libs/glagol_sdk/i_discovery.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <atomic>
#include <cstdint>

namespace YandexIO {
    class DeviceContext {
    public:
        using onConnectedCb = std::function<void()>;
        using onRegistrationResult = std::function<void(RegistrationResult /* result */)>;

        /**
         * @brief DeviceContext is an abstraction btw YandexIOSDK and Quasar services. YandexIOSDK communicate with
         * DeviceContext through "yandexio" IPC server. DeviceContext has an IPC connector to "yandexio" and handles
         * ioControl messages. At the same time DeviceContext sends ioEvent messages to "yandexio" server (each
         * fire* method send QuasarMessage with ioEvent)
         * @param onConnected - callback that will be called when DeviceContext internal IPC connector  will actually connect
         * @param autoConnect - true - device context will connect to "yandexio" automatically;
         *                      false - need to call connectToSDK manually. It can be used if it's important to set
         *                              up all callbacks before connecting
         * connect to "yandexio" IPC server (it means that vendor called YandexIOSDK::start method)
         */
        DeviceContext(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory, onConnectedCb onConnected = nullptr, bool autoConnect = true);

        DeviceContext(const DeviceContext&) = delete;
        DeviceContext(DeviceContext&&) = delete;

    public:
        /**
         * @brief Manually connect to IO SDK server
         */
        void connectToSDK();

        /**
         * @brief Synchronously shutdown connection to yandexio
         */
        void shutdown();

        void setSetupMode(bool isSetup, bool isFirstSetup = false);

        void fireSoundDataTransferStart();
        void fireConnectingToNetwork();
        void fireSetupError();

        void fireConversationError();

        void fireEnqueueAlarm(const quasar::proto::IOEvent_AlarmType& alarmType, const std::string& alarmId);
        void fireStartAlarm(const quasar::proto::IOEvent_AlarmType& alarmType);
        void fireStopAlarm(const quasar::proto::IOEvent_AlarmType& alarmType, bool hasRemainingMedia);
        void fireAlarmStopRemainingMedia();
        void fireAlarmsSettingsChanged(const quasar::proto::AlarmsSettings& alarmsSettings);

        void fireMediaRequest(quasar::proto::MediaContentType contentType);
        void fireMediaStarted(quasar::proto::MediaContentType contentType);
        void fireMediaPaused(quasar::proto::MediaContentType contentType);
        void fireMediaResumed(quasar::proto::MediaContentType contentType);
        void fireMediaError(quasar::proto::MediaContentType contentType);

        void fireMediaSwitchedForward(quasar::proto::MediaContentType contentType);
        void fireMediaSwitchedBackward(quasar::proto::MediaContentType contentType);
        void fireMediaLiked(quasar::proto::MediaContentType contentType);
        void fireMediaDisliked(quasar::proto::MediaContentType contentType);

        void fireNotificationPending();
        void fireNotificationStarted();
        void fireNotificationEnd();

        void fireBtAvrcpPause();
        void fireBtAvrcpResume();
        void fireBtAvrcpNext();
        void fireBtAvrcpPrev(bool forced);

        void fireBtTakeAudioFocus();
        void fireBtFreeAudioFocus();

        /**
         * @brief Notify YandexIOSDK users that authentication succeeded or failed
         * @param oauthCode -- oauth code, which was used in authentication
         * @param is_ok -- was it successful
         * @param response -- response message. It's in JSON format, it has two fields:
         * 1. addUserStatus -- string. It represents result of getting token from oauthCode. Has 3 options: OK, CODE_EXPIRED, NO_INTERNET.
         * 2. registerResponse -- JSON. Response of register request. If addUserStatus is not OK, this field make no sense.
         */
        void fireAuthenticationStatus(const std::string& oauthCode, bool is_ok, const std::string& response);

        /**
         * @brief Notify SDK user that oauth token is invalid and it's time to renew it
         */
        void fireOAuthTokenIsInvalid(const std::string& oauthToken);

        /**
         * @brief Notify SDK user that current authentication is invalid (probably has been revoked by user)
         */
        void fireAuthenticationIsInvalid(const std::string& uid);

        /**
         * @brief Send Last update info on device boot up
         * @param wasAttempted - false - there wasn't any update (manual start). true - device tried to apply update
         * @param attemptSucceeded - false - couldn't apply OTA update. true - device successfully updated
         */
        void fireLastUpdateState(bool wasAttempted, bool attemptSucceeded);

        void fireExternalCommand(const std::string& name, const std::string& vinsRequestId, const std::string& payload);

        void fireSystemConfig(const std::string& configName, const std::string& jsonConfigValue);
        void fireDeviceConfig(const std::string& configName, const std::string& jsonConfigValue);
        void fireAccountConfig(const std::string& configName, const std::string& jsonConfigValue);

        void firePushNotification(const std::string& operation, const std::string& messageStr);

        void fireSDKState(const quasar::proto::IOEvent::SDKState& state);

        void fireSpectrum(quasar::proto::IOEvent::Spectrum&& spectrum);

        void fireSoundCommand(const std::string& command);

        void fireStartingConfigureSuccess(bool shouldUpdate);

        void fireMusicStateChanged(const quasar::proto::AppState::MusicState& musicState);

        void fireAudioClientEvent(const quasar::proto::AudioClientEvent& audioClientEvent);

        void fireBrickStatusChanged(const quasar::proto::BrickInfo& brickInfo);

        void fireDeviceGroupStateChanged(const quasar::proto::DeviceGroupState& groupState) const;

        void fireDiscoveryResult(const glagol::IDiscovery::Result& discoveryResult) const;

        void waitUntilConnected() const;

        /**
         * @brief isHttpRequestOk - true when get_sync_info http request processed successfully
         *                          false - when network error occured
         */
        void fireGetSyncInfoStatus(bool isHttpRequestOk);

        std::string preprocessVinsResponse(const std::string& response);

        std::function<void()> onAllowInitConfigurationState;

        std::function<void(const std::string& /* oauth token*/, const std::string& /* uid */,
                           onRegistrationResult /* onResult */)>
            onRegistrationInBackendRequested;

        /**
         * @brief Provide User account info from trusted source (expect that oauthToken and uid matches)
         * @note: If token and uid are empty -- user account info is revoked
         */
        std::function<void(const std::string& /* oauth token*/, const std::string& /* uid */)> onUserAccountInfo;

        /**
         * @brief Execute vins response as if it comes from the backend
         */
        std::function<void(std::string)> onVinsResponse;

        std::function<void(bool)> onSetupMode;
        /**
         * @brief Callback to start/stop setup mode. Internal quasar handler will decide
         * to stop or start start up mode
         */
        std::function<void()> onToggleSetupMode;

        std::function<void(const quasar::proto::BluetoothSinkEvent& sinkEvent)> onBluetoothSinkMediaEvent;

        std::function<void()> onPreparedForNotification;

        /**
         * @brief Play user sound file (should be located in /system/vendor/quasar/sounds/)
         * @param soundFile - .mp3/.wav filename to play
         * @param repeatCount - how many times file will be played
         * @param priority - request priority for playbackd
         * @param shouldCancelOnListening - should cancel play on listening
         */
        std::function<void(const std::string& soundFile, int repeatCount, int priority, bool shouldCancelOnListening)> onPlaySound;
        /**
         *  @brief Plays user sound file (should be located in /system/vendor/quasar/sounds/) in playing group:
         *         only one track can be played in certain group at a time, but tracks in different groups play concurrently
         *  @param groupName - group name
         *  @param soundFile - .mp3/.wav filename to play
         *  @param repeatCount - how many times file will be played
         */
        std::function<void(const std::string& groupName, const std::string& soundFile, int repeatCount)> onPlayPlayingGroup;
        /**
         * @brief Cancel user playing group
         * @param group - group name
         * @param dropPlayer - destroy player
         */
        std::function<void(const std::string& group, bool dropPlayer)> onCancelSound;

        std::function<void(const std::string& alarmId)> onApproveAlarm;

        /**
         * @brief subscribes to certain device config from backend
         */
        std::function<void(const std::string&)> onSubscribeToDeviceConfig;
        /**
         * @brief unsubscribes from certain device config from backend
         */
        std::function<void(const std::string&)> onUnsubscribeFromDeviceConfig;

        /**
         * @brief subscribes to certain system config from backend
         */
        std::function<void(const std::string&)> onSubscribeToSystemConfig;
        /**
         * @brief unsubscribes from certain system config from backend
         */
        std::function<void(const std::string&)> onUnsubscribeFromSystemConfig;

        /**
         * @brief subscribes to certain account config from backend
         */
        std::function<void(const std::string&)> onSubscribeToAccountConfig;
        /**
         * @brief unsubscribes from certain account config from backend
         */
        std::function<void(const std::string&)> onUnsubscribeFromAccountConfig;

        std::function<void(const std::string& oauthToken)> onChangeAccount;

        /**
         * @brief allows or prohibits update downloading. When it's needed: must be used when device has it's own battery
         * @param allowUpdateForAll -- bool value, if it's true, we can download all updates (including crit)
         * @param allowUpdateForCrit -- bool value, if it's true, we can download crit updates (even if allowUpdateForAll is false)
         */
        std::function<void(bool allowUpdateForAll, bool allowUpdateForCrit)> onAllowUpdate;

        std::function<void(TvPolicyInfo tvPolicyInfo)> onTvPolicyInfoReceived;
        std::function<void(const NAlice::TDeviceState::TActiveActions& activeActions)> onActiveActionReceived;
        std::function<void(const std::optional<std::string>& activeActionPayload)> onActiveActionSemanticFrameReceived;

        /**
         * @brief Gives numeric metric to monitord, allows to handle platform-specific metrics like internal
         */
        std::function<void(const std::string& key, double value)> onNumericMetrics;
        /**
         * @brief Gives categorical metric to monitord, allows to handle platform-specific metrics like internal
         */
        std::function<void(const std::string& key, const std::string& value)> onCategoricalMetrics;

        /**
         * @brief Alice interaction blocking
         */
        std::function<void(const std::string& source, const std::optional<std::string>& soundError)> onAssistantBlocking;

        /**
         * @brief Alice interaction unblocking
         */
        std::function<void(const std::string& source)> onAssistantUnblocking;

        std::function<void()> onAcceptIncomingCall;
        std::function<void()> onDeclineIncomingCall;
        std::function<void()> onDeclineCurrentCall;

        /**
         * @brief Fires up when interface report screen active state
         */
        std::function<void(bool isScreenActive)> onIsScreenActive;

        std::function<void(const yandex_io::proto::TDeviceStatePart& statePart)> onDeviceStatePart;

        /**
         * @brief Timezone was set
         */
        std::function<void(const quasar::proto::Timezone& timezone)> onTimezone;

        /**
         * @brief Location was set
         */
        std::function<void(const quasar::proto::Location& location)> onLocation;

        /**
         * @brief WifiList was set
         */
        std::function<void(const quasar::proto::WifiList& wifiList)> onWifiList;

        /**
         * @brief MediaServiceParams was set
         */
        std::function<void(const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier)> onMediaDeviceIdentifier;

        std::function<void(ClockDisplayState)> onClockDisplayStateReceived;

        std::function<void(const quasar::proto::EqualizerConfig& config)> onEqualizerConfig;

    private:
        void fireStartSetup(bool firstSetup); // DeviceContext users should use setSetupMode() function.
        void fireFinishSetup();               // DeviceContext users should use setSetupMode() function.

        void handleConnect();

        void handleQuasarMessage(const quasar::ipc::SharedMessage& message);

        std::shared_ptr<quasar::ipc::IConnector> connector_;

        bool onAllowInitConfigurationStateDone_ = false;

        /**
         * @brief onConnected is a callback that will be called each time when DeviceContext connects to "yandexio"
         * server (aka YandexIOSDK). So it can be used to notify YandexIO about device state (i.e. sdk is ready)
         */
        onConnectedCb onConnected_;
    };

} // namespace YandexIO
