#include "capability_proxy.h"

#include <yandex_io/sdk/private/endpoint_storage/converters/remote_object_id_factory.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/remoting_message_builder.h>
#include <yandex_io/sdk/private/remoting/i_remoting_registry.h>

#include <util/system/yassert.h>

namespace YandexIO {

    CapabilityProxy::CapabilityProxy(
        std::string id,
        NAlice::TCapabilityHolder state,
        std::shared_ptr<DirectiveHandlerProxy> directiveHandler,
        std::weak_ptr<IRemotingRegistry> remotingRegistry,
        std::shared_ptr<IConnectionRegistry> connectionRegistry)
        : IRemoteObject(std::move(remotingRegistry))
        , id_(std::move(id))
        , connections_(std::move(connectionRegistry))
        , state_(std::move(state))
        , directiveHandler_(std::move(directiveHandler))
    {
    }

    CapabilityProxy::~CapabilityProxy()
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            if (!remoteObjectId_.empty()) {
                remotingRegistry->removeRemoteObject(remoteObjectId_);
            }
        }
    }

    void CapabilityProxy::init(const std::string& endpointId)
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            auto remoteObjectId = RemoteObjectIdFactory::createId(endpointId, shared_from_this());
            if (remotingRegistry->addRemoteObject(remoteObjectId, weak_from_this())) {
                remoteObjectId_ = std::move(remoteObjectId);
            }
        }
    }

    void CapabilityProxy::setState(NAlice::TCapabilityHolder state)
    {
        state_ = std::move(state);

        for (const auto& wlistener : listeners_) {
            if (auto slistener = wlistener.lock()) {
                slistener->onCapabilityStateChanged(shared_from_this(), state_);
            }
        }
    }

    void CapabilityProxy::fireEvents(const std::vector<NAlice::TCapabilityEvent>& events) {
        for (const auto& wlistener : listeners_) {
            if (auto slistener = wlistener.lock()) {
                slistener->onCapabilityEvents(shared_from_this(), events);
            }
        }
    }

    const std::string& CapabilityProxy::getId() const {
        return id_;
    }

    NAlice::TCapabilityHolder CapabilityProxy::getState() const {
        return state_;
    }

    IDirectiveHandlerPtr CapabilityProxy::getDirectiveHandler() {
        return directiveHandler_;
    }

    void CapabilityProxy::addListener(std::weak_ptr<IListener> wlistener)
    {
        listeners_.push_back(wlistener);
    }

    void CapabilityProxy::removeListener(std::weak_ptr<IListener> wlistener)
    {
        const auto iter = std::find_if(listeners_.begin(), listeners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });
        if (iter != listeners_.end()) {
            listeners_.erase(iter);
        }
    }

    void CapabilityProxy::handleRemotingMessage(
        const quasar::proto::Remoting& message,
        std::shared_ptr<IRemotingConnection> connection)
    {
        const auto connectionStorageName = connections_->getStorageName(connection);
        if (!connectionStorageName.has_value()) {
            return;
        }

        if (message.has_capability_state_changed_method()) {
            const auto& method = message.capability_state_changed_method();
            setState(method.capability().capability());
        } else if (message.has_capability_events_method()) {
            const auto& method = message.capability_events_method();

            std::vector<NAlice::TCapabilityEvent> events;
            events.reserve(method.events().size());

            for (const auto& event : method.events()) {
                events.push_back(event);
            }

            fireEvents(events);
        }

        // broadcast event to other proxies
        connections_->broadcastMessage(message, {*connectionStorageName});
    }

    ProxyCapabilityListener::ProxyCapabilityListener(std::string endpointId,
                                                     std::shared_ptr<IConnectionRegistry> connections)
        : endpointId_(std::move(endpointId))
        , connections_(std::move(connections))
    {
        Y_VERIFY(!endpointId_.empty());
        Y_VERIFY(connections_);
    }

    void ProxyCapabilityListener::onCapabilityStateChanged(const std::shared_ptr<ICapability>& capability,
                                                           const NAlice::TCapabilityHolder& state) {
        const auto remoting = RemotingMessageBuilder::buildCapabilityStateChanged(endpointId_, capability, state);
        connections_->broadcastMessage(remoting, {});
    }

    void ProxyCapabilityListener::onCapabilityEvents(const std::shared_ptr<ICapability>& capability,
                                                     const std::vector<NAlice::TCapabilityEvent>& events) {
        const auto remoting = RemotingMessageBuilder::buildCapabilityEvents(endpointId_, capability, events);
        connections_->broadcastMessage(remoting, {});
    }

} // namespace YandexIO
