#include "connection_registry.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <sstream>

namespace YandexIO {
    IConnectionRegistry::~IConnectionRegistry() = default;

    void ConnectionRegistry::putConnection(const std::string& name, std::weak_ptr<IRemotingConnection> connection) {
        connections_[name] = std::move(connection);
    }

    void ConnectionRegistry::broadcastMessage(const quasar::proto::Remoting& message, const std::set<std::string>& ignoredConnections) {
        if (YIO_LOG_DEBUG_ENABLED()) {
            std::stringstream ss;
            ss << "BroadcastMessage. Connections: ";
            for (const auto& [name, con] : connections_) {
                ss << name << ',';
            }
            ss << "; IgnoredConnections: ";
            for (const auto& name : ignoredConnections) {
                ss << name << ',';
            }
            ss << " Message: " << quasar::shortUtf8DebugString(message);
            YIO_LOG_DEBUG(ss.str());
        }
        for (const auto& [name, wconnection] : connections_) {
            if (ignoredConnections.contains(name)) {
                continue;
            }
            if (const auto connection = wconnection.lock()) {
                connection->sendMessage(message);
            }
        }
    }

    void ConnectionRegistry::sendMessage(const quasar::proto::Remoting& message, const std::string& storageName) {
        YIO_LOG_DEBUG("SendMessage to: " << storageName << " message: " << quasar::shortUtf8DebugString(message));
        if (const auto it = connections_.find(storageName); it != connections_.end()) {
            if (auto connection = it->second.lock()) {
                connection->sendMessage(message);
            }
        }
    }

    size_t ConnectionRegistry::getSize() const {
        return connections_.size();
    }

    std::optional<std::string> ConnectionRegistry::getStorageName(const std::weak_ptr<IRemotingConnection>& connection) const {
        for (const auto& [name, wconnection] : connections_) {
            if (!connection.owner_before(wconnection) && !wconnection.owner_before(connection)) {
                return name;
            }
        }
        return std::nullopt;
    }

    std::shared_ptr<IRemotingConnection> ConnectionRegistry::getConnection(const std::string& storageName) const {
        if (const auto it = connections_.find(storageName); it != connections_.end()) {
            const auto& wconnection = it->second;
            return wconnection.lock();
        }
        return nullptr;
    }

} // namespace YandexIO
