#include "converters.h"

#include <yandex_io/libs/logging/logging.h>
#include <google/protobuf/any.pb.h>

namespace YandexIO {

    NAlice::TEndpoint convertEndpointStateToProtobuf(
        const std::shared_ptr<IEndpoint>& endpoint)
    {
        NAlice::TEndpoint protobuf;

        protobuf.set_id(TString(endpoint->getId()));
        protobuf.mutable_meta()->set_type(endpoint->getState().meta().type());
        protobuf.mutable_meta()->mutable_deviceinfo()->CopyFrom(endpoint->getState().meta().deviceinfo());
        protobuf.mutable_status()->CopyFrom(endpoint->getState().status());

        return protobuf;
    }

    quasar::proto::Capability convertCapabilityToProtobuf(
        const std::shared_ptr<IEndpoint>& endpoint,
        const std::shared_ptr<ICapability>& capability)
    {
        quasar::proto::Capability protobuf;

        protobuf.set_id(TString(capability->getId()));
        protobuf.mutable_capability()->CopyFrom(capability->getState());

        if (const auto& directiveHandler = capability->getDirectiveHandler()) {
            protobuf.mutable_directive_handler()->CopyFrom(convertDirectiveHandlerToProtobuf(endpoint->getId(), directiveHandler));
        }

        return protobuf;
    }

    quasar::proto::DirectiveHandler convertDirectiveHandlerToProtobuf(
        const std::string& endpointId,
        const std::shared_ptr<IDirectiveHandler>& directiveHandler)
    {
        quasar::proto::DirectiveHandler protobuf;

        protobuf.set_endpoint_id(TString(endpointId));
        protobuf.set_name(TString(directiveHandler->getHandlerName()));
        for (const auto& name : directiveHandler->getSupportedDirectiveNames()) {
            protobuf.add_supported_directive_names(TString(name));
        }

        return protobuf;
    }

    void appendCapabilities(NAlice::TEndpoint& state, std::list<std::shared_ptr<ICapability>> capabilities) {
        for (const auto& capability : capabilities) {
            try {
                const auto capabilityState = capability->getState();
                const auto& message = ICapability::getCapabilityFromHolder(capabilityState);
                state.add_capabilities()->PackFrom(message);
            } catch (const std::invalid_argument& e) {
                YIO_LOG_WARN("Failed to add capability: " << e.what());
            }
        }
    }

} // namespace YandexIO
