#include "remoting_message_builder.h"
#include "converters.h"
#include "remote_object_id_factory.h"

#include <yandex_io/libs/base/utils.h>

namespace YandexIO {

    quasar::proto::Remoting RemotingMessageBuilder::buildAddEndpoint(const std::shared_ptr<IEndpointStorage>& storage,
                                                                     const std::string& storageName,
                                                                     const quasar::proto::Endpoint& endpoint) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(storage);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_storage_method();
        method->set_storage_name(TString(storageName));
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointStorageMethod::ADD_ENDPOINT);
        method->add_endpoints()->CopyFrom(endpoint);

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildRemoveEndpoint(const std::shared_ptr<IEndpointStorage>& storage,
                                                                        const std::string& storageName,
                                                                        const quasar::proto::Endpoint& endpoint) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(storage);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_storage_method();
        method->set_storage_name(TString(storageName));
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointStorageMethod::REMOVE_ENDPOINT);
        method->add_endpoints()->CopyFrom(endpoint);

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildSyncEndpoints(const std::shared_ptr<IEndpointStorage>& storage,
                                                                       const std::string& storageName,
                                                                       const std::vector<quasar::proto::Endpoint>& endpoints) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(storage);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_storage_method();
        method->set_storage_name(TString(storageName));
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointStorageMethod::SYNC_ENDPOINTS);
        for (const auto& endpoint : endpoints) {
            method->add_endpoints()->CopyFrom(endpoint);
        }

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::builCapabilityConfig(const std::shared_ptr<IEndpointStorage>& storage,
                                                                         const quasar::proto::CapabilityConfig& config) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(storage);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_storage_send_capability_config_method();
        method->mutable_capability_config()->CopyFrom(config);

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildAddCapability(const std::shared_ptr<IEndpoint>& endpoint,
                                                                       const std::shared_ptr<ICapability>& capability,
                                                                       const std::string& storageName) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(endpoint);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_method();
        method->set_storage_name(TString(storageName));
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointMethod::ADD_CAPABILITY);
        method->mutable_capability()->CopyFrom(convertCapabilityToProtobuf(endpoint, capability));

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildRemoveCapability(const std::shared_ptr<IEndpoint>& endpoint,
                                                                          const std::shared_ptr<ICapability>& capability) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(endpoint);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_method();
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointMethod::REMOVE_CAPABILITY);
        method->mutable_capability()->CopyFrom(convertCapabilityToProtobuf(endpoint, capability));

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildEndpointSyncState(const std::shared_ptr<IEndpoint>& endpoint) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(endpoint);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_endpoint_method();
        method->set_id(quasar::makeUUID());
        method->set_method(quasar::proto::Remoting::EndpointMethod::SYNC_STATE);
        method->mutable_state()->CopyFrom(convertEndpointStateToProtobuf(endpoint));

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildCapabilityStateChanged(const std::string& endpointId,
                                                                                const std::shared_ptr<ICapability>& capability,
                                                                                const NAlice::TCapabilityHolder& state) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(endpointId, capability);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_capability_state_changed_method();
        method->mutable_capability()->set_id(TString(capability->getId()));
        method->mutable_capability()->mutable_capability()->CopyFrom(state);

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildCapabilityEvents(const std::string& endpointId,
                                                                          const std::shared_ptr<ICapability>& capability,
                                                                          const std::vector<NAlice::TCapabilityEvent>& events) {
        quasar::proto::Remoting message;
        const auto remoteObjectId = RemoteObjectIdFactory::createId(endpointId, capability);
        message.set_remote_object_id(TString(remoteObjectId));

        auto method = message.mutable_capability_events_method();
        for (const auto& event : events) {
            method->add_events()->CopyFrom(event);
        }

        return message;
    }

    quasar::proto::Remoting RemotingMessageBuilder::buildDirectiveHandlerMethod(const std::string& remoteObjectId,
                                                                                const std::shared_ptr<IDirectiveHandler>& directiveHandler,
                                                                                const std::shared_ptr<Directive>& directive,
                                                                                quasar::proto::Remoting::DirectiveHandlerMethod::Method method) {
        quasar::proto::Remoting remoting;
        remoting.set_remote_object_id(TString(remoteObjectId));

        auto handlerMethod = remoting.mutable_directive_handler_method();
        handlerMethod->set_method(method);
        handlerMethod->set_endpoint_id(TString(directiveHandler->getEndpointId()));
        handlerMethod->set_handler_name(TString(directiveHandler->getHandlerName()));
        handlerMethod->mutable_directive()->CopyFrom(Directive::convertToDirectiveProtobuf(directive));

        return remoting;
    }

} // namespace YandexIO
