#include "endpoint.h"
#include "proxy_factory.h"

#include <yandex_io/sdk/private/remoting/i_remoting_registry.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/converters.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/remote_object_id_factory.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/remoting_message_builder.h>

#include <yandex_io/libs/logging/logging.h>

#include <util/generic/yexception.h>

YIO_DEFINE_LOG_MODULE("endpoint_proxy");

namespace YandexIO {

    Endpoint::Endpoint(std::string id,
                       NAlice::TEndpoint::EEndpointType type,
                       NAlice::TEndpoint::TDeviceInfo deviceInfo,
                       std::shared_ptr<IDirectiveHandler> directiveHandler,
                       std::string storageName,
                       bool isHost,
                       std::string hostStorageName,
                       std::weak_ptr<IRemotingRegistry> remotingRegistry,
                       std::shared_ptr<IConnectionRegistry> connectionRegistry)
        : EndpointBase(std::move(id), type, std::move(deviceInfo), std::move(directiveHandler))
        , IRemoteObject(std::move(remotingRegistry))
        , storageName_(std::move(storageName))
        , hostNodeConnectionName_(std::move(hostStorageName))
        , isHost_(isHost)
        , connections_(std::move(connectionRegistry))
    {
    }

    Endpoint::~Endpoint()
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            if (!remoteObjectId_.empty()) {
                remotingRegistry->removeRemoteObject(remoteObjectId_);
            }
        }
    }

    std::list<std::shared_ptr<ICapability>> Endpoint::getCapabilities() const {
        std::list<std::shared_ptr<ICapability>> res;
        for (const auto& [storageName, capabilities] : capabilities_) {
            for (const auto& capability : capabilities) {
                res.push_back(capability);
            }
        }
        return res;
    }

    void Endpoint::addCapability(const std::shared_ptr<ICapability>& capability) {
        addCapabilityImpl(storageName_, capability);

        const auto proxyListener = std::make_shared<ProxyCapabilityListener>(getId(), connections_);
        proxyListenerByCapId_[capability->getId()] = proxyListener;
        capability->addListener(proxyListener);

        const auto remoting = RemotingMessageBuilder::buildAddCapability(shared_from_this(), capability, storageName_);
        if (isHost_) {
            connections_->broadcastMessage(remoting, {});
        } else {
            outgoingRequests_.insert(remoting.endpoint_method().id());
            connections_->sendMessage(remoting, hostNodeConnectionName_);
        }
    }

    void Endpoint::removeCapability(const std::shared_ptr<ICapability>& capability) {
        removeProxyListenerForCapability(capability);
        removeCapabilityImpl(capability);

        const auto remoting = RemotingMessageBuilder::buildRemoveCapability(shared_from_this(), capability);
        if (isHost_) {
            connections_->broadcastMessage(remoting, {});
        } else {
            outgoingRequests_.insert(remoting.endpoint_method().id());
            connections_->sendMessage(remoting, hostNodeConnectionName_);
        }
    }

    void Endpoint::setStatus(const NAlice::TEndpoint::TStatus& status) {
        // Note: Only Endpoint owner can update it's status
        if (!isHost_) {
            return;
        }
        EndpointBase::setStatus(status);
        const auto remoting = RemotingMessageBuilder::buildEndpointSyncState(shared_from_this());
        connections_->broadcastMessage(remoting, {});
    }

    void Endpoint::initRemoting()
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            auto remoteObjectId = RemoteObjectIdFactory::createId(shared_from_this());
            if (remotingRegistry->addRemoteObject(remoteObjectId, weak_from_this())) {
                remoteObjectId_ = std::move(remoteObjectId);
            }
        }
    }

    void Endpoint::initProxyCapabilities(const quasar::proto::Endpoint& message)
    {
        for (const auto& holder : message.capabilities()) {
            Y_ENSURE(!holder.storage_name().empty());
            Y_ENSURE(holder.has_capability());
            const auto capability = YandexIO::createCapabilityProxy(holder.capability(),
                                                                    getRemotingRegistry(),
                                                                    remoteObjectId_,
                                                                    connections_,
                                                                    hostNodeConnectionName_);
            capability->init(getId());
            addCapabilityImpl(holder.storage_name(), capability);
        }
    }

    void Endpoint::handleRemotingMessage(
        const quasar::proto::Remoting& message,
        std::shared_ptr<IRemotingConnection> connection)
    {
        // Directive Handler Method handle
        // Handle it directly. If Endpoint contains CapabilityProxy -> DirectveHandlerProxy
        // will redirect this message itself
        if (message.has_directive_handler_method()) {
            handleDirectiveHandlerMethod(message.directive_handler_method());
            return;
        }

        // Endpoint method handle
        if (!message.has_endpoint_method() || message.endpoint_method().id().empty()) {
            return;
        }
        const auto connectionStorageName = connections_->getStorageName(connection);
        if (!connectionStorageName.has_value()) {
            return;
        }
        const auto& method = message.endpoint_method();
        const std::string& reqid = method.id();

        if (const auto it = outgoingRequests_.find(reqid); it != outgoingRequests_.end()) {
            // Request was scheduled by this Endpoint instance. Ignore incoming message.
            // Consider it as a success response from EndpointHost
            outgoingRequests_.erase(it);
            return;
        }
        // Check if message should be routed to EndpointHost
        if (!isHost_ && *connectionStorageName != hostNodeConnectionName_) {
            // route message to Endpoint host
            routedReqIdToConnection_[reqid] = *connectionStorageName;
            connections_->sendMessage(message, hostNodeConnectionName_);
            return;
        }

        switch (method.method()) {
            case quasar::proto::Remoting::EndpointMethod::ADD_CAPABILITY: {
                if (isHost_) {
                    handleRemoteAddCapability(method, *connectionStorageName);
                } else {
                    if (const auto it = routedReqIdToConnection_.find(reqid); it != routedReqIdToConnection_.end()) {
                        // EndpointProxy requested ADD_CAPABILITY. EndpointHost created it
                        // Create DirectiveHandlerProxy with direct connection to CapabilityHost (inside EndpointProxy)
                        const auto& connectionName = it->second;
                        handleRemoteAddCapability(method, connectionName);
                    } else {
                        // EndpointHost added CapabilityHost. Create DirectiveHandlerProxy with direct connection to host
                        handleRemoteAddCapability(method, *connectionStorageName);
                    }
                }
                break;
            }
            case quasar::proto::Remoting::EndpointMethod::REMOVE_CAPABILITY: {
                handleRemoteRemoveCapability(method);
                break;
            }
            case quasar::proto::Remoting::EndpointMethod::SYNC_STATE: {
                handleRemoteSyncState(method);
                break;
            }
            default: {
                YIO_LOG_ERROR_EVENT("Endpoint.FailedToHandleEndpointMethod", "Unsupported method: " << method.method());
                break;
            }
        }

        // notify other proxies
        std::set<std::string> ignoredConnections;
        if (*connectionStorageName == hostNodeConnectionName_) {
            // do not send same message to EndpointHost
            ignoredConnections.insert(hostNodeConnectionName_);
        }
        routedReqIdToConnection_.erase(reqid);
        connections_->broadcastMessage(message, ignoredConnections);
    }

    void Endpoint::handleRemoteAddCapability(const quasar::proto::Remoting::EndpointMethod& method,
                                             const std::string& connectionStorageName) {
        Y_ENSURE(!method.storage_name().empty());
        auto capability = YandexIO::createCapabilityProxy(method.capability(), getRemotingRegistry(), remoteObjectId_, connections_, connectionStorageName);
        capability->init(getId());
        addCapabilityImpl(method.storage_name(), capability);
    }

    void Endpoint::handleRemoteRemoveCapability(const quasar::proto::Remoting::EndpointMethod& method) {
        if (auto capability = findCapabilityById(method.capability().id())) {
            removeProxyListenerForCapability(capability);
            removeCapabilityImpl(capability);
        }
    }

    void Endpoint::handleRemoteSyncState(const quasar::proto::Remoting::EndpointMethod& method) {
        // NOTE: Only host can update status of Endpoint
        if (!isHost_) {
            EndpointBase::setStatus(method.state().status());
        }
    }

    void Endpoint::handleDirectiveHandlerMethod(const quasar::proto::Remoting::DirectiveHandlerMethod& method) {
        if (auto directiveHandler = findDirectiveHandler(method.handler_name())) {
            auto directive = Directive::createDirectiveFromProtobuf(method.directive());
            switch (method.method()) {
                case quasar::proto::Remoting::DirectiveHandlerMethod::HANDLE: {
                    directiveHandler->handleDirective(directive);
                    break;
                }
                case quasar::proto::Remoting::DirectiveHandlerMethod::CANCEL: {
                    directiveHandler->cancelDirective(directive);
                    break;
                }
                case quasar::proto::Remoting::DirectiveHandlerMethod::PREFETCH: {
                    directiveHandler->prefetchDirective(directive);
                    break;
                }
            }
        } else {
            YIO_LOG_WARN("Can't find directive handler: " << method.handler_name());
        }
    }

    std::shared_ptr<IDirectiveHandler> Endpoint::findDirectiveHandler(const std::string& handlerName) const {
        if (auto directiveHandler = getDirectiveHandler(); directiveHandler && directiveHandler->getHandlerName() == handlerName) {
            return directiveHandler;
        }
        for (const auto& capability : getCapabilities()) {
            if (auto directiveHandler = capability->getDirectiveHandler()) {
                if (directiveHandler->getHandlerName() == handlerName) {
                    return directiveHandler;
                }
            }
        }
        return nullptr;
    }

    void Endpoint::dropProxyCapabilities() {
        for (auto& [storageName, capabilities] : capabilities_) {
            if (storageName == storageName_) {
                continue;
            }
            for (auto it = capabilities.begin(); it != capabilities.end();) {
                const auto& capability = *it;
                const auto sthis = shared_from_this();
                forEachListener([&sthis, &capability](const auto& listener) {
                    listener->onCapabilityRemoved(sthis, capability);
                });
                it = capabilities.erase(it);
            }
        }
    }

    void Endpoint::addCapabilityImpl(const std::string& storageName, const std::shared_ptr<ICapability>& capability) {
        // todo: ensure unique
        capabilities_[storageName].push_back(capability);
        forEachListener([this, &capability](const auto& listener) {
            listener->onCapabilityAdded(shared_from_this(), capability);
        });
    }

    void Endpoint::removeCapabilityImpl(const std::shared_ptr<ICapability>& capability) {
        for (auto& [storageName, capabilities] : capabilities_) {
            const auto it = std::find(capabilities.begin(), capabilities.end(), capability);
            if (it == capabilities.end()) {
                continue;
            }
            capabilities.erase(it);
            forEachListener([this, &capability](const auto& listener) {
                listener->onCapabilityRemoved(shared_from_this(), capability);
            });
            break;
        }
    }

    void Endpoint::removeProxyListenerForCapability(const std::shared_ptr<ICapability>& capability) {
        if (const auto it = proxyListenerByCapId_.find(capability->getId()); it != proxyListenerByCapId_.end()) {
            const auto& proxyListener = it->second;
            capability->removeListener(proxyListener);
            proxyListenerByCapId_.erase(it);
        }
    }

    std::shared_ptr<IEndpoint> Endpoint::sharedFromThis()
    {
        return shared_from_this();
    }

    quasar::proto::Endpoint Endpoint::toProto() {
        quasar::proto::Endpoint protobuf;

        const auto sthis = sharedFromThis();
        protobuf.mutable_endpoint()->CopyFrom(convertEndpointStateToProtobuf(sthis));

        for (const auto& [storage, capabilities] : capabilities_) {
            for (const auto& capability : capabilities) {
                quasar::proto::Endpoint::CapabilityHolder holder;
                holder.mutable_capability()->CopyFrom(convertCapabilityToProtobuf(sthis, capability));
                holder.set_storage_name(TString(storage));
                protobuf.add_capabilities()->Swap(&holder);
            }
        }

        if (const auto directiveHandler = getDirectiveHandler()) {
            auto convertedDirectiveHandler = convertDirectiveHandlerToProtobuf(getId(), directiveHandler);
            protobuf.mutable_directive_handler()->Swap(&convertedDirectiveHandler);
        }
        return protobuf;
    }

} // namespace YandexIO
