#include "endpoint_base.h"

#include <yandex_io/sdk/private/endpoint_storage/converters/converters.h>
#include <yandex_io/libs/logging/logging.h>

#include <google/protobuf/any.pb.h>

namespace YandexIO {

    EndpointBase::EndpointBase(std::string id, NAlice::TEndpoint::EEndpointType type, NAlice::TEndpoint::TDeviceInfo deviceInfo, std::shared_ptr<IDirectiveHandler> directiveHandler)
        : id_(std::move(id))
        , type_(type)
        , deviceInfo_(std::move(deviceInfo))
        , directiveHandler_(std::move(directiveHandler))
    {
    }

    const std::string& EndpointBase::getId() const {
        return id_;
    }

    NAlice::TEndpoint EndpointBase::getState() const {
        NAlice::TEndpoint state;

        state.set_id(TString(id_));
        state.mutable_meta()->set_type(type_);
        state.mutable_meta()->mutable_deviceinfo()->CopyFrom(deviceInfo_);
        state.mutable_status()->CopyFrom(status_);
        appendCapabilities(state, getCapabilities());
        return state;
    }

    void EndpointBase::setStatus(const NAlice::TEndpoint::TStatus& status)
    {
        status_ = status;
        forEachListener([this](const auto& listener) {
            listener->onEndpointStateChanged(sharedFromThis());
        });
    }

    NAlice::TEndpoint::TStatus EndpointBase::getStatus() const {
        return status_;
    }

    void EndpointBase::addListener(std::weak_ptr<IListener> wlistener)
    {
        wlisteners_.push_back(wlistener);
    }

    void EndpointBase::removeListener(std::weak_ptr<IListener> wlistener)
    {
        const auto iter = std::find_if(wlisteners_.begin(), wlisteners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });

        if (iter != wlisteners_.end()) {
            wlisteners_.erase(iter);
        }
    }

    std::shared_ptr<IDirectiveHandler> EndpointBase::getDirectiveHandler() const {
        return directiveHandler_;
    }

    void EndpointBase::forEachListener(const std::function<void(const std::shared_ptr<IListener>)>& callback) {
        for (const auto& wlistener : wlisteners_) {
            if (const auto slistener = wlistener.lock()) {
                callback(slistener);
            }
        }
    }

} // namespace YandexIO
